#!/usr/bin/perl -w
#
# install Netop Guest found in current directory.
# (c) Netop Business Solutions A/S 2016
#

my $VERSION = "install: 1.09";
my $DEBUG = 0;
my $VERBOSE = 1;

my $MIN_SUPPORTED_OS_VER = 16;
my $MAX_SUPPORTED_OS_VER = 18;

use Cwd;
use File::Path;
use File::Copy;
use File::Basename;

use lib '.';

my $SCRIPTDIR=dirname(__FILE__);
my $CRTDIR=cwd();

my $NETOPGUEST_BIN = "netopguest";
my $NETOPGUEST = "/opt/netop/guest/bin/" . $NETOPGUEST_BIN;
# change product version here, it will be seen by each <package>inst.pm
my $NETOP_PRODUCT_VERSION = '13.06';
my $GUEST_LICENSE_DIR = "/var/opt/netop/guest";
my $GUEST_LICENSE_FILE = $GUEST_LICENSE_DIR."/guest.lic";



sub dir_is_empty {
	my $dir_name = shift;
	opendir(my $dir_handle, $dir_name) or return 0;
	my $entries_num = scalar(grep {$_ ne "." && $_ ne ".."} readdir($dir_handle));
	closedir($dir_handle);
	return ($entries_num == 0);
}

sub dbg {
	if($DEBUG) {
		my $arg = shift(@_);
		print STDERR "DBG: $arg";
	}
}

sub su_test {
	if(! -e '/usr/bin/id') {
		return 1; # continue, cannot get uid.
	}
	my $id = `/usr/bin/id`;
	if($id =~ m|uid=([0-9]+)|) {
		if($1 eq "0") {
			return 1;
		}
	}
	return 0;
}

# $_[0] is product name to display (MANDATORY)
# $_[1] is the validator subroutine passed as a reference (MANDATORY) that must return a boolean flag.
# $_[2] is the patience ( times to ask ) (OPTIONAL) = 3
# $_[3] is an optional argument to the validator (OPTIONAL) = undef
sub patiently_ask_for_serial {
	( $product_name, $validator, $patience, $validator_arg ) = @_;
	$answer = undef;

	return ( 0, $answer ) unless ( ( defined $product_name ) and ( defined $validator ) );
	$patience = 1 * $patience; # force numeric context
	$patience = 3 unless ( ( defined $patience ) and ( $patience ) ); # default patience : perl is patient
	while ( $patience )	{
		my $options = "[T]rial / [p]ortal / [c]ustom";

		print "\n[$patience] Which type of license do you want to use for $product_name? $options: ";
		$answer = <STDIN>;
		
		my $lower = lc $answer;
		if ( ($lower eq "p\n")) { 
			$answer = "PORTAL"; 
		}
		elsif (($answer eq "\n") || ($lower eq "t\n")) { 
			$answer = "TRIAL"; 
		}
		elsif (($lower eq "c\n")) { 
			print "\n\t[$patience] Enter License key for $product_name ( or leave blank for trial ):\n";
			$answer = <STDIN>;

			if ( (!$answer) or ($answer =~ /^\s*$/) )  { 
				$answer = "TRIAL"; 
			} 
		}
		else {    
			print "Please reply: $options\n";
			next;  
		}

		if ( $validator->( $answer, $validator_arg ) ) { 
			return ( $patience, $answer ); 
		}
		$patience--;
		print "\nThe license key could not be validated, please try again. ($patience)\n";
	}
	print STDERR "\nNone of the license keys could be validated.\n";
	return ( 0, $answer );
} # sub patiently_ask_for_serial


$OS_VER = "";
$OS_BIT = "";
$SYSTEM = "";
$OS_ID = "";

sub system_setup {
	my $uname = `uname`;
	dbg("uname  = $uname");
	if ($uname =~ "Darwin") {
		$SYSTEM = "Darwin";
		$OS_VER = `uname -r`;
	}
	else {
		# get os name vers arch		
		my $osdetails = `cat /etc/*-release`;
		if ($osdetails =~ m|\bNAME="(.*)"|) {
			$SYSTEM = $1;
		}
		elsif ($osdetails =~ m|\bNAME=(.*)|) {
			$SYSTEM = $1;
		}
		if ($osdetails =~ m|\bID="(.*)"|) {
			$OS_ID = $1;
		}
		elsif ($osdetails =~ m|\bID=(.*)|) {
			$OS_ID = $1;
		}
		if ($osdetails =~ m|\bVERSION_ID="(\d{1,2}).(\d{1,2})"|) {
			$OS_VER = $1;
		}
		elsif ($osdetails =~ m|\bVERSION_ID=(\d{1,2}).(\d{1,2})|) {
			$OS_VER = $1;
		}
		elsif ($osdetails =~ m|\bVERSION_ID="(\d{1,2})"|) {
			$OS_VER = $1;
		}
		elsif ($osdetails =~ m|\bVERSION_ID=(\d{1,2})|) {
			$OS_VER = $1;
		}
		$OS_BIT = `uname -m`;
	}
	dbg("SYSTEM = $SYSTEM\n");
	dbg("OS_ID  = $OS_ID\n");
	dbg("OS_VER = $OS_VER\n");
	dbg("OS_BIT = $OS_BIT\n");
}

sub input_confirmation_N {
	my $options = "[yes/No]";

	my $question = shift(@_);
	print "$question $options >";
	while(<STDIN>) {
		my $lower = lc $_;
		if( ($lower =~ ("y"))) {
			return "y";
		} 
		elsif (($_ eq "\n") || ($lower =~ ("n"))) {
			return "n";
		}  
		else {
			print "Please reply: $options\n";
			print "$question $options >";
		}
	}
}

# moved validation here :
# expects serial number as an argument 
sub validate_guest_serial_number {
	my $serial = shift(@_);
	return 0 unless defined $serial;
	my $command = "$NETOPGUEST -no_splash -serialno $serial";
	my $status = system($command);
	if ( 0 == $status )
	{
		print "\n\tSerial Number validated OK.\n";
		return 1;
	}
	else
	{
		print STDERR "\nSerial Number is INVALID.\n";
		return 0;
	}
} # sub validate_guest_serial_number

sub guest_installed_but_not_validated {
	if (-f $GUEST_LICENSE_FILE && -s $GUEST_LICENSE_FILE) {
                print "\nThe Netop Guest application has been installed.\n";
                print "Netop Guest will try to use the existig license file ($GUEST_LICENSE_FILE)\n";
	} 
	else {
                print "\nThe Netop Guest application has been installed, but not validated.\n";
		print "Please run '$NETOPGUEST_BIN -serialno <your license key>' as super-user to validate the product.\n";
                print "Afterwards, you can start the Netop Guest application by typing '$NETOPGUEST_BIN' as a regular user.\n";
	}
        print "Thank you for using Netop Business Solutions A/S!\n\n";
} # sub guest_installed_but_not_validated

sub guest_installed_and_validated {
        print "\nNetop Guest application has been installed and activated.\n";
        print "You can start the Netop Guest application by typing '$NETOPGUEST_BIN' as a regular user.\n";
        print "Thank you for using Netop Business Solutions A/S!\n\n";
} # sub guest_installed_and_validated

# $_[0] is a flag that tells wether to allow user to scroll or not ( OPTIONAL )
sub print_license {
	my $scroll_method = "more";
	my $allow_scroll = shift;
	$allow_scroll = 0 + $allow_scroll; # force numeric ctx
	if ( ! defined $allow_scroll ) { 
		$allow_scroll = 1; 
	}
	my $command = "cat $SCRIPTDIR/eula.txt";
	if ( 1 == $allow_scroll ) { 
		$command = $command . " | $scroll_method"; 
	}
	system($command);
} # sub print_license


sub migrate_guest_license() {
	if (-f "/etc/NetopGuest/guest.lic" && ! -f "$GUEST_LICENSE_FILE") {
		mkpath($GUEST_LICENSE_DIR) unless (-e $GUEST_LICENSE_DIR);
		system("/bin/mv /etc/NetopGuest/guest.lic $GUEST_LICENSE_FILE");
	}
	if (-d "/etc/NetopGuest/" && dir_is_empty("/etc/NetopGuest/")) {
		rmdir("/etc/NetopGuest/");
	}
}

sub input_confirmation_Y
{
   my $options = "[Yes/no]";

   my $question = shift(@_);
   print "$question $options >";
   while(<STDIN>) {
       my $lower = lc $_;
       if( ($lower =~ ("n"))) {
           return "n";
       } elsif (($_ eq "\n") || ($lower =~ ("y"))) {
           return "y";
       } else {
       print "Please reply: $options\n";
           print "$question $options >";
       }
   }
}

sub main() {
	my $autoinstall = 0;

	my $guest_serial = "";
	my $host_serial = "";
	my $oldserial   = "";
	my $usage =  "Usage: install.pl <param>\n"
	." where <param> is:\n"
	."    [--help]           : print help message and exit.\n"
	."    [--version]        : print version info and exit.\n"
        ."    [--license]        : print the Netop License.\n"
	."    [--debug]          : turn debug on.\n"
        ."    [--serial <serial>] : install Netop Guest with <serial> number license.\n"
        ."    [--autoinstall]    : non-interactive install, assumes you agree with the Netop License.\n";

	while($arg = shift @ARGV) {
		if ($arg eq "--version") {
			print "$VERSION\n";
			exit;
		} 
		elsif ( $arg eq "--license" ) {
			print_license( 1 );
			exit ( 0 );
		} 
		elsif ($arg eq "--debug") {
			$DEBUG = 1;
		} 
		elsif ($arg eq "--autoinstall") {
			$autoinstall = 1;
		} 
		elsif ($arg eq "--help") {
			print $usage;
			exit;
		} 
		elsif ($arg eq "--serial") {
			# next token is not another option ?
			if ( ( defined $ARGV[0] ) and ( "--" ne substr($ARGV[0],0,2)) ) { $guest_serial = shift @ARGV; } #
		} 
		else {
			print "Error: unknown arg: '$arg'\n";
			die $usage;
		}
	}

	die "Must have super-user privileges to install this application.\nPlease elevate your privileges with 'su -' and try again.\n" if(! su_test());

	system_setup();
	
    # determine if deb-files available
        my $debfiles = `/bin/ls netop*.deb 2>/dev/null`;
	$debfiles =~ s/\n//g;
	if($debfiles eq "") {
		die "DEB package not found\n";
	}
        my $debfile = `/bin/ls netop-guest_*.deb`;
    $debfile =~ s/\n//g;
    die "Guest package version $NETOP_PRODUCT_VERSION not found\n" if(!($debfile =~ m|.*$NETOP_PRODUCT_VERSION|));

	my $install_dir = `pwd`;
	$install_dir =~ s/\n//g;

    print "\nInstalling Netop Guest $NETOP_PRODUCT_VERSION\n";
    # need to make a distinction between install failure and user
    # choosing N(o) in interactive CLI :
    my $result = 0;
	
	# verify if the current OS is supported, if not, give a warning
	my $supported = 1;
	if ($OS_ID =~ "ubuntu") {
		if ($OS_VER gt $MAX_SUPPORTED_OS_VER || $OS_VER lt $MIN_SUPPORTED_OS_VER) {
			$supported = 0;
		}
	}
	else { #debian ?
		$supported = 0;
	}
	if (!$supported) {
		print "Detected OS $SYSTEM $OS_VER, which is not officially supported by this package.\n";
		if (!$autoinstall) {
			my $answer = input_confirmation_Y("Do you still want to continue with the installation?");
			$answer = lc substr($answer,0,1);
			if ( 'y' ne $answer )
			{
				exit;
			}
		}
		else {
			print "Will attempt to continue because of --autoinstall flag.\n";
		}
	}

	if ( $autoinstall ) {
		if ( $guest_serial eq "" ) {
			print "Note: you did not provide any license (--serial <serial>). A trial license will be used.\n";
        }
	}
	
	my $deb_ca_cert = "/etc/ssl/certs/ca-certificates.crt";
	my $rh_ca_cert = "/etc/pki/tls/certs/ca-bundle.crt";
	my $sles_ca_cert = "/etc/ssl/ca-bundle.pem";

	my $default_ca_cert = "ca-certificates.crt";
	my $ca_cert_dir = "/var/opt/netop/certs/";
	my $ca_cert_file = "/var/opt/netop/certs/ca-bundle.crt";
	
	#make sure the ca certificate is available in certificates directory
	print "Installing CA certificates bundle in $ca_cert_dir\n";

	unless (-e $ca_cert_file) {
		eval { mkpath($ca_cert_dir) unless (-e $ca_cert_dir) };
		if ($@) {
			print "Couldn't create $ca_cert_dir: $@\n";
		}
		if (-e $deb_ca_cert) {
			symlink($deb_ca_cert, $ca_cert_file);
		}
		elsif (-e $rh_ca_cert) {
			symlink($rh_ca_cert, $ca_cert_file);
		}
		elsif (-e $sles_ca_cert) {
			symlink($sles_ca_cert, $ca_cert_file);
		}
		else {
			die "No CA certificate" unless (-e $default_ca_cert);
			copy($default_ca_cert,  $ca_cert_file);
		}    
	}

	# query for already installed product (netop package name format)
	my $cmd = "dpkg-query -W netop-guest 2>/dev/null";
	my $installed = `$cmd`;
	$installed =~ s/\n//g;
	#print "Query Package = $installed\n";
	if($installed =~ m{netop-guest}) {
		#print "Package installed!\n";
		dbg("Netop Guest already installed : '$installed' !\n");
		$cmd = "dpkg -r netop-guest";
		my $res = system("$cmd");
		if($res != 0) {
			print STDERR "Install Guest: '$cmd' failed with status $res, trying to upgrade.\n";
		}
	}
	#else {
	#	print "Package '$installed' , not installed!\n";
	#}

    my $pkgname = "netop-guest";
    my $dpkg_command = "/usr/bin/dpkg -i $debfile 2>&1 1> /dev/null";
    my $apt_command = "/usr/bin/apt-get -y -f install 2>&1 1> /dev/null";
	if ($DEBUG == 1) {
		$dpkg_command = "/usr/bin/dpkg -i $debfile";
		$apt_command = "/usr/bin/apt-get -y -f install";
	}
	dbg("running command: $dpkg_command\n");
    my $command = `$dpkg_command`;
    my $dpkg_result = $?;
    my $apt_result = $?;
    if ( 0 != $dpkg_result ) {
		dbg("dpkg_result = $dpkg_result\n");
		# attempt to correct system
		dbg("running command: $apt_command\n");
        $command = `$apt_command`;
        $apt_result = $?;
        if( 0 != $apt_result) {
			# install had anyhow failed
            dbg("apt_result = $apt_result\n");
			dbg("running command: /usr/bin/dpkg --purge $pkgname $pkgname:i386\n");
			if ($DEBUG == 1) {
				$apt_result = `/usr/bin/dpkg --purge $pkgname $pkgname:i386`;
			}
			else {
				$apt_result = `/usr/bin/dpkg --purge $pkgname $pkgname:i386 2>&1 1> /dev/null`;
			}
			$result = 0;
        }
		else {
			$result = 1;
		}
    }
    else {
    	$result = 1;
    }
	dbg("result = $result\n");
    # installation failed ?
    if ( 1 != $result ) {
        # 'hard' failure of installer :
        if ( 0 == $result ) {
            print STDERR "\nInstallation FAILED!.\n";
            print STDERR "Install command exited with result $dpkg_result\n";
            exit;
        }
        # user chose N(o) (or C(ancel)) in package manager :
        if ( 2 == $result ) {
            print STDERR "You chose not to install this package on this system.\n";
            print STDERR "Installer can not continue for this product.\n";
            exit;
        }
    }
    migrate_guest_license();
        
    print "\nThe following license-agreement apply to Netop Guest:\n";
    print "[Please read carefully]\n";
    my $res = '';

    if ( 0 == $autoinstall ) {
        print_license( ! $autoinstall );
        $res = input_confirmation_N("Do you accept this license?");
        $res = substr($res,0,1);
        if ( 'y' ne $res ) {
            print "License Agreement not accepted, program will not be installed!\n";
            exit;
        }
    }
    else {
        print_license( ! $autoinstall );
        print "Auto-accepted license because of --autoinstall flag.\n";
    }
    if( -e "license.dat") {
        my $license = `cat license.dat`;
        print "The following trial license key apply to Netop Guest:\n";
        print "$license\n\n";
    }
    if ( "" eq $guest_serial ) {
        if ( !$autoinstall ) {
            ( $validated, $serial ) = patiently_ask_for_serial("Netop Guest", \&validate_guest_serial_number, 3, undef );
            if ( $validated ) {
                if (0) { 
                    print "\t Validated with serial : $serial\n"; 
                }
                guest_installed_and_validated();
                exit;
            }
        }
		else {
			$validated = validate_guest_serial_number("TRIAL");
			if ( $validated ) {
				print "\t Validated with trial license.\n"; 
				guest_installed_and_validated();
                exit;
			}
		}
        guest_installed_but_not_validated();
        exit;
    } # if ( !$guest_serial )
    else {
        my $result = validate_guest_serial_number( $guest_serial );
        if ( 1 == $result ) {
            guest_installed_and_validated();
            exit;
        }
        guest_installed_but_not_validated();
    } # else ( !$guest_serial )
}

main;
exit;
__END__
