#!/usr/bin/perl -w
#
# install Netop Host found in current directory.
# (c) Netop Business Solutions A/S 2016
#

my $VERSION = "install: 1.09";
my $DEBUG = 0;
my $VERBOSE = 1;

my $MIN_SUPPORTED_OS_VER = 10;
my $MAX_SUPPORTED_OS_VER = 10;

use Cwd;
use File::Path;
use File::Copy;
use File::Basename;

use lib '.';

my $SCRIPTDIR=dirname(__FILE__);
my $CRTDIR=cwd();
# change product version here, it will be seen by each <package>inst.pm
my $NETOP_PRODUCT_VERSION = '13.06';

my $NETOPHOST_ROOT_DIR = "/opt/netop/host";
my $NETOPHOST_BIN = "netophost";
my $NETOPHOST = $NETOPHOST_ROOT_DIR."/bin/".$NETOPHOST_BIN;

my $NETOPHOST_VAR_ROOT_DIR = "/var/opt/netop/host";

my $HOST_INIFILE = "/var/opt/netop/host/host.xml";


sub dir_is_empty {
	my $dir_name = shift;
	opendir(my $dir_handle, $dir_name) or return 0;
	my $entries_num = scalar(grep {$_ ne "." && $_ ne ".."} readdir($dir_handle));
	closedir($dir_handle);
	return ($entries_num == 0);
}

sub convert_host_files {
	my $old_host_cfg_dir = "/etc/NetopHost";
	
	if ( -f "$old_host_cfg_dir/NetopHost.xml" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.xml") {
		system("/bin/mv $old_host_cfg_dir/NetopHost.xml $NETOPHOST_VAR_ROOT_DIR/host.xml");
	}
	if ( -f "$old_host_cfg_dir/NetopHost.dwm" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.dwm") {
		system("/bin/mv $old_host_cfg_dir/NetopHost.dwm $NETOPHOST_VAR_ROOT_DIR/host.dwm");
	}
	if (-f "$old_host_cfg_dir/host.xml" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.xml") {
		system("/bin/mv $old_host_cfg_dir/host.xml $NETOPHOST_VAR_ROOT_DIR/host.xml");
	}
	if (-f "$old_host_cfg_dir/license.dat" && ! -f "$NETOPHOST_VAR_ROOT_DIR/license.dat") {
		system("/bin/mv $old_host_cfg_dir/license.dat $NETOPHOST_VAR_ROOT_DIR/license.dat");
	}
	if (-f "$old_host_cfg_dir/host.lic" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.lic") {
		system("/bin/mv $old_host_cfg_dir/host.lic $NETOPHOST_VAR_ROOT_DIR/host.lic");
	}
	if (-f "$old_host_cfg_dir/host.uuid" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.uuid") {
		system("/bin/mv $old_host_cfg_dir/host.uuid $NETOPHOST_VAR_ROOT_DIR/host.uuid");
	}
	if (-f "$old_host_cfg_dir/host.uuid_old" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.uuid_old") {
		system("/bin/mv $old_host_cfg_dir/host.uuid_old $NETOPHOST_VAR_ROOT_DIR/host.uuid_old");
	}
	
	if (-d $old_host_cfg_dir && dir_is_empty($old_host_cfg_dir)) {
		rmdir($old_host_cfg_dir);
	}
	
	if (-f "/etc/netop.ini" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.ini") {
		system("/bin/cp /etc/netop.ini $NETOPHOST_VAR_ROOT_DIR/host.ini");
	}
}

sub dbg {
	if($DEBUG) {
		my $arg = shift(@_);
		print STDERR "DBG: $arg";
	}
}

sub su_test {
	if(! -e '/usr/bin/id') {
		return 1; # continue, cannot get uid.
	}
	my $id = `/usr/bin/id`;
	if($id =~ m|uid=([0-9]+)|) {
		if($1 eq "0") {
			return 1;
		}
	}
	return 0;
}

$OS_VER = "";
$OS_BIT = "";
$SYSTEM = "";
$OS_ID = "";

sub system_setup {
	my $uname = `uname`;
	dbg("uname  = $uname");
	if ($uname =~ "Darwin") {
		$SYSTEM = "Darwin";
		$OS_VER = `uname -r`;
	}
	else {
		# get os name vers arch		
		my $osdetails = `cat /etc/*-release`;
		if ($osdetails =~ m|\bNAME="(.*)"|) {
			$SYSTEM = $1;
		}
		elsif ($osdetails =~ m|\bNAME=(.*)|) {
			$SYSTEM = $1;
		}
		if ($osdetails =~ m|\bID="(.*)"|) {
			$OS_ID = $1;
		}
		elsif ($osdetails =~ m|\bID=(.*)|) {
			$OS_ID = $1;
		}
		if ($osdetails =~ m|\bVERSION_ID="(\d{1,2}).(\d{1,2})"|) {
			$OS_VER = $1;
		}
		elsif ($osdetails =~ m|\bVERSION_ID=(\d{1,2}).(\d{1,2})|) {
			$OS_VER = $1;
		}
		elsif ($osdetails =~ m|\bVERSION_ID="(\d{1,2})"|) {
			$OS_VER = $1;
		}
		elsif ($osdetails =~ m|\bVERSION_ID=(\d{1,2})|) {
			$OS_VER = $1;
		}
		$OS_BIT = `uname -m`;
	}
	dbg("SYSTEM = $SYSTEM\n");
	dbg("OS_ID  = $OS_ID\n");
	dbg("OS_VER = $OS_VER\n");
	dbg("OS_BIT = $OS_BIT\n");
}

# $_[0] is a flag that tells wether to allow user to scroll or not ( OPTIONAL )
sub print_license {
	my $scroll_method = "more";
	my $allow_scroll = shift;
	$allow_scroll = 0 + $allow_scroll; # force numeric ctx
	if ( ! defined $allow_scroll ) { 
		$allow_scroll = 1; 
	}
	my $command = "cat $SCRIPTDIR/eula.txt";
	if ( 1 == $allow_scroll ) { 
		$command = $command . " | $scroll_method"; 
	}
	system($command);
} # sub print_license

sub input_confirmation_Y
{
   my $options = "[Yes/no]";

   my $question = shift(@_);
   print "$question $options >";
   while(<STDIN>) {
       my $lower = lc $_;
       if( ($lower =~ ("n"))) {
           return "n";
       } elsif (($_ eq "\n") || ($lower =~ ("y"))) {
           return "y";
       } else {
       print "Please reply: $options\n";
           print "$question $options >";
       }
   }
}

sub main() {
	my $autoinstall_flag = "";
	my $autoinstall = 0;
	my $host_serial = "";
	my $usage =  "Usage: install.pl <param>\n"
	." where <param> is:\n"
	."    [--help]           : print help message and exit.\n"
	."    [--version]        : print version info and exit.\n"
        ."    [--license]        : print the Netop License.\n"
    ."    [--debug]          : turn debug on.\n"
        ."    [--serial <serial>]         : install Netop Host with <serial> number license.\n"
        ."    [--autoinstall]    : non-interactive install, assumes you agree with the Netop License.\n";

	while($arg = shift @ARGV) {
		if ($arg eq "--version") {
			print "$VERSION\n";
			exit;
		} 
		elsif ( $arg eq "--license" ) {
			print_license( 1 );
			exit ( 0 );
		} 
		elsif ($arg eq "--debug") {
			$DEBUG = 1;
		} 
		elsif ($arg eq "--autoinstall") {
        	$autoinstall_flag = "--autoinstall";
			$autoinstall = 1;
		} 
		elsif ($arg eq "--help") {
			print $usage;
			exit;
		} 
		elsif ($arg eq "--serial") {
			# next token must not be another option 
			if ( ( defined $ARGV[0] ) and ( "--" ne substr($ARGV[0],0,2) ) ) { 
				$host_serial = shift @ARGV;
			}
			else {
				die "bad command line";
			}
		}
		else {
			print "Error: unknown arg: '$arg'\n";
			die $usage;
		}
	}

	die "Must have super-user privileges to install this application.\nPlease elevate your privileges with 'su -' and try again.\n"
		if(! su_test());
		
	system_setup();

	# determine if deb-file available
        my $debfiles = `/bin/ls netop*.deb 2>/dev/null`;
	$debfiles =~ s/\n//g;
	if($debfiles eq "") {
		die "DEB package not found\n";
	}
	# try to get 32 and 64 bit packages
        my $deb64 = `/bin/ls netop-host*_amd64.deb 2> /dev/null`;
	$deb64 =~ s/\n//g;
        my $deb32 = `/bin/ls netop-host*_i386.deb 2> /dev/null`;
	$deb32 =~ s/\n//g;


	# see if using right package
	chomp (my $arch = `uname -m`);
	my $bIs64 = $arch eq "x86_64";
	my $debfile = "";
	
	if ($bIs64) {
                if ( $deb64 =~ m|.*$NETOP_PRODUCT_VERSION| ) {
                        $debfile = $deb64;
                } 
                elsif ( $deb32 =~ m|.*$NETOP_PRODUCT_VERSION| ) {
                        print "You are installing a 32-bit package on a 64-bit system. There is also a 64-bit package available, that you can download and use.\n";
                        if (!$autoinstall) {
                                my $answer = input_confirmation_Y("Do you still want to continue with this installation?");
                                $answer = lc substr($answer,0,1);
                                if ( 'y' ne $answer )
                                {
                                        exit;
                                }
                        }
                        else {
                                print "Will continue because of --autoinstall flag.\n";
                        }
                        $debfile = $deb32;
                } 
                else {
                        die "Host package version $NETOP_PRODUCT_VERSION not found\n";
                }				
        }
        else { # 32-bit system
                if ( $deb64 =~ m|.*$NETOP_PRODUCT_VERSION| ) {
                        die "You are attempting to install a 64-bit package on a 32-bit system. Please download and install the 32-bit package instead.\n";
                } 
                elsif ( $deb32 =~ m|.*$NETOP_PRODUCT_VERSION| ) {
                        $debfile = $deb32;
                } 
                else {
                        die "Host package version $NETOP_PRODUCT_VERSION not found\n";
                }
        }

	my $install_dir = `pwd`;
	$install_dir =~ s/\n//g;

	# check if the configuration file already exists
	my $config_file_exists = 0;
	if (-e $HOST_INIFILE) {
			$config_file_exists = 1;
	}

        print "\nInstalling Netop Host $NETOP_PRODUCT_VERSION\n";
	my $result = 0;
	
	# verify if the current OS is supported, if not, give a warning
	my $supported = 1;
	if ($OS_ID =~ "debian") {
		if ($OS_VER gt $MAX_SUPPORTED_OS_VER || $OS_VER lt $MIN_SUPPORTED_OS_VER) {
			$supported = 0;
		}
	}
	else { #ubuntu ?
		$supported = 0;
	}
	if (!$supported) {
		print "Detected OS $SYSTEM $OS_VER, which is not officially supported by this package.\n";
		if (!$autoinstall) {
			my $answer = input_confirmation_Y("Do you still want to continue with the installation?");
			$answer = lc substr($answer,0,1);
			if ( 'y' ne $answer )
			{
				exit;
			}
		}
		else {
			print "Will attempt to continue because of --autoinstall flag.\n";
		}
	}


	if ( $autoinstall ) {
		if ( $host_serial eq "" ) {
			print "Note: you did not provide any license (--serial <serial>). A trial license will be used.\n";
        }
	}

	my $deb_ca_cert = "/etc/ssl/certs/ca-certificates.crt";
	my $rh_ca_cert = "/etc/pki/tls/certs/ca-bundle.crt";
	my $sles_ca_cert = "/etc/ssl/ca-bundle.pem";

	my $default_ca_cert = "ca-certificates.crt";
	my $ca_cert_dir = "/var/opt/netop/certs/";
	my $ca_cert_file = "/var/opt/netop/certs/ca-bundle.crt";
	
	#make sure the ca certificate is available in certificates directory
	print "Installing CA certificates bundle in $ca_cert_dir\n";

	unless (-e $ca_cert_file) {
		eval { mkpath($ca_cert_dir) unless (-e $ca_cert_dir) };
		if ($@) {
			print "Couldn't create $ca_cert_dir: $@\n";
		}
		if (-e $deb_ca_cert) {
			symlink($deb_ca_cert, $ca_cert_file);
		}
		elsif (-e $rh_ca_cert) {
			symlink($rh_ca_cert, $ca_cert_file);
		}
		elsif (-e $sles_ca_cert) {
			symlink($sles_ca_cert, $ca_cert_file);
		}
		else {
			die "No CA certificate" unless (-e $default_ca_cert);
			copy($default_ca_cert,  $ca_cert_file);
		}    
	}

	# query for already installed product (netop package name format)
	my $cmd = "dpkg-query -W netop-host 2>/dev/null";
	my $installed = `$cmd`;
	$installed =~ s/\n//g;
	#print "Query Package = $installed\n";
	if($installed =~ m{netop-host}) {
		#print "Package installed!\n";
		dbg("Netop Host already installed : '$installed' !\n");
		$cmd = "/etc/init.d/netophostd stop >/dev/null 2>/dev/null";
		$cmd = "dpkg -r netop-host";
		my $res = system("$cmd");
		if($res != 0) {
			print STDERR "Install Host: '$cmd' failed with status $res, trying to upgrade.\n";
		}
	}
	#else {
	#	print "Package '$installed' , not installed!\n";
	#}

        my $pkgname = "netop-host";
	my $dpkg_command = "/usr/bin/dpkg -i $debfile 2>&1 1> /dev/null";
	my $apt_command = "/usr/bin/apt-get -y -f install 2>&1 1> /dev/null";
	if ($DEBUG == 1) {
		$dpkg_command = "/usr/bin/dpkg -i $debfile";
		$apt_command = "/usr/bin/apt-get -y -f install";
	}
	dbg("running command: $dpkg_command\n");
	my $res = `$dpkg_command`;
	my $dpkg_result = $?;
	my $apt_result = $?;

	if ( 0 != $dpkg_result ) {
		dbg("dpkg_result = $dpkg_result\n");
		# attempt to correct system
		dbg("running command: $apt_command\n");
		$res = `$apt_command`;
		$apt_result = $?;
		if( 0 != $apt_result) {
			# install had anyhow failed
			dbg("apt_result = $apt_result\n");
			dbg("running command: /usr/bin/dpkg --purge $pkgname $pkgname:i386\n");
			if ($DEBUG == 1) {
				$apt_result = `/usr/bin/dpkg --purge $pkgname $pkgname:i386`;
			}
			else {
				$apt_result = `/usr/bin/dpkg --purge $pkgname $pkgname:i386 2>&1 1> /dev/null`;
			}
			$result = 0;
		}
		else {
			$result = 1;
		}
	}
	else {
		$result = 1;
	}
	dbg("result = $result\n");

	# installation failed :
	if ( 1 != $result ) {
		# 'hard' failure of installer :
		if ( 0 == $result ) {
			print STDERR "\nInstallation FAILED!.\n";
			print STDERR "Install command exited with result $dpkg_result\n";
			exit( 1 );
		}
		# user chose N(o) in gdebi interactive install :
		if ( 2 == $result ) {
			print STDERR "You chose not to install this package on this system.\n";
			print STDERR "Installer can not continue for this product.\n";
			exit( 1 );
		}
	}
	
	convert_host_files();
	
	my $configure_script = "/opt/netop/host/configure/configure.pl";
	if (! -e $configure_script) {
                print "Netop Host installed but not yet configured for use\n";
		exit;
	}
	
	$cmd = "perl $configure_script --serial $host_serial $autoinstall_flag";
	if ($config_file_exists == 1) {
		$cmd = $cmd . " --upgrade"
	}
	dbg("cmd: $cmd\n");
	my $st = system($cmd);
	$st >>= 8;
	dbg("configure.pl st: $st\n");
	if ($st == 0) {
	} 
	else {
		if ($st == 56) {
			$method = "/opt/netop/host/configure/uninstall.pl";
                        print "\t(run '$method' to uninstall Netop Host)\n"; # TODO
		}
	}

}

main;
exit;
__END__
