#!/usr/bin/perl -w
#
# install Netop Host rpm found in current directory.
# (c) Netop Business Solutions A/S 2016
#

my $VERSION = "install: 1.09";
my $DEBUG = 0;
my $VERBOSE = 1;

my $MIN_SUPPORTED_OS_VER = 12;
my $MAX_SUPPORTED_OS_VER = 12;

use Cwd;
use File::Path;
use File::Copy;
use File::Basename;

use lib '.';

my $SCRIPTDIR=dirname(__FILE__);
my $CRTDIR=cwd();
# change product version here, it will be seen by each <package>inst.pm
my $NETOP_PRODUCT_VERSION = '13.06';

my $NETOPHOST_ROOT_DIR = "/opt/netop/host";
my $NETOPHOST_BIN = "netophost";
my $NETOPHOST = $NETOPHOST_ROOT_DIR."/bin/".$NETOPHOST_BIN;

my $NETOPHOST_VAR_ROOT_DIR = "/var/opt/netop/host";

my $HOST_INIFILE = "/var/opt/netop/host/host.xml";


sub dir_is_empty {
	my $dir_name = shift;
	opendir(my $dir_handle, $dir_name) or return 0;
	my $entries_num = scalar(grep {$_ ne "." && $_ ne ".."} readdir($dir_handle));
	closedir($dir_handle);
	return ($entries_num == 0);
}

sub convert_host_files {
	my $old_host_cfg_dir = "/etc/NetopHost";
	
	if ( -f "$old_host_cfg_dir/NetopHost.xml" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.xml") {
		system("/bin/mv $old_host_cfg_dir/NetopHost.xml $NETOPHOST_VAR_ROOT_DIR/host.xml");
	}
	if ( -f "$old_host_cfg_dir/NetopHost.dwm" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.dwm") {
		system("/bin/mv $old_host_cfg_dir/NetopHost.dwm $NETOPHOST_VAR_ROOT_DIR/host.dwm");
	}
	if (-f "$old_host_cfg_dir/host.xml" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.xml") {
		system("/bin/mv $old_host_cfg_dir/host.xml $NETOPHOST_VAR_ROOT_DIR/host.xml");
	}
	if (-f "$old_host_cfg_dir/license.dat" && ! -f "$NETOPHOST_VAR_ROOT_DIR/license.dat") {
		system("/bin/mv $old_host_cfg_dir/license.dat $NETOPHOST_VAR_ROOT_DIR/license.dat");
	}
	if (-f "$old_host_cfg_dir/host.lic" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.lic") {
		system("/bin/mv $old_host_cfg_dir/host.lic $NETOPHOST_VAR_ROOT_DIR/host.lic");
	}
	if (-f "$old_host_cfg_dir/host.uuid" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.uuid") {
		system("/bin/mv $old_host_cfg_dir/host.uuid $NETOPHOST_VAR_ROOT_DIR/host.uuid");
	}
	if (-f "$old_host_cfg_dir/host.uuid_old" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.uuid_old") {
		system("/bin/mv $old_host_cfg_dir/host.uuid_old $NETOPHOST_VAR_ROOT_DIR/host.uuid_old");
	}
	
	if (-d $old_host_cfg_dir && dir_is_empty($old_host_cfg_dir)) {
		rmdir($old_host_cfg_dir);
	}
	
	if (-f "/etc/netop.ini" && ! -f "$NETOPHOST_VAR_ROOT_DIR/host.ini") {
		system("/bin/cp /etc/netop.ini $NETOPHOST_VAR_ROOT_DIR/host.ini");
	}
}

sub dbg {
	if($DEBUG) {
		my $arg = shift(@_);
		print STDERR "DBG: $arg";
	}
}

sub su_test {
	if(! -e '/usr/bin/id') {
		return 1; # continue, cannot get uid.
	}
	my $id = `/usr/bin/id`;
	if($id =~ m|uid=([0-9]+)|) {
		if($1 eq "0") {
			return 1;
		}
	}
	return 0;
}

$OS_VER = "";
$OS_BIT = "";
$SYSTEM = "";
$OS_ID = "";

sub system_setup {
	my $uname = `uname`;
	dbg("uname  = $uname");
	if ($uname =~ "Darwin") {
		$SYSTEM = "Darwin";
		$OS_VER = `uname -r`;
	}
	else {
		# get os name vers arch		
		my $osdetails = `cat /etc/*-release`;
		if ($osdetails =~ m|\bNAME="(.*)"|) {
			$SYSTEM = $1;
		}
		elsif ($osdetails =~ m|\bNAME=(.*)|) {
			$SYSTEM = $1;
		}
		if ($osdetails =~ m|\bID="(.*)"|) {
			$OS_ID = $1;
		}
		elsif ($osdetails =~ m|\bID=(.*)|) {
			$OS_ID = $1;
		}
		if ($osdetails =~ m|\bVERSION_ID="(\d{1,2}).(\d{1,2})"|) {
			$OS_VER = $1;
		}
		elsif ($osdetails =~ m|\bVERSION_ID=(\d{1,2}).(\d{1,2})|) {
			$OS_VER = $1;
		}
		elsif ($osdetails =~ m|\bVERSION_ID="(\d{1,2})"|) {
			$OS_VER = $1;
		}
		elsif ($osdetails =~ m|\bVERSION_ID=(\d{1,2})|) {
			$OS_VER = $1;
		}
		$OS_BIT = `uname -m`;
	}
	dbg("SYSTEM = $SYSTEM\n");
	dbg("OS_ID  = $OS_ID\n");
	dbg("OS_VER = $OS_VER\n");
	dbg("OS_BIT = $OS_BIT\n");
}

# $_[0] is a flag that tells wether to allow user to scroll or not ( OPTIONAL )
sub print_license {
	my $scroll_method = "more";
	my $allow_scroll = shift;
	$allow_scroll = 0 + $allow_scroll; # force numeric ctx
	if ( ! defined $allow_scroll ) { 
		$allow_scroll = 1; 
	}
	my $command = "cat $SCRIPTDIR/eula.txt";
	if ( 1 == $allow_scroll ) { 
		$command = $command . " | $scroll_method"; 
	}
	system($command);
} # sub print_license

sub input_confirmation_Y
{
   my $options = "[Yes/no]";

   my $question = shift(@_);
   print "$question $options >";
   while(<STDIN>) {
       my $lower = lc $_;
       if( ($lower =~ ("n"))) {
           return "n";
       } elsif (($_ eq "\n") || ($lower =~ ("y"))) {
           return "y";
       } else {
       print "Please reply: $options\n";
           print "$question $options >";
       }
   }
}

sub main() {
	my $autoinstall_flag = "";
	my $autoinstall = 0;
	my $host_serial = "";
	my $usage =  "Usage: install.pl <param>\n"
	." where <param> is:\n"
	."    [--help]           : print help message and exit.\n"
	."    [--version]        : print version info and exit.\n"
        ."    [--license]        : print the Netop License.\n"
	."    [--debug]          : turn debug on.\n"
        ."    [--serial <serial>]         : install Netop Host with <serial> number license.\n"
        ."    [--autoinstall]    : non-interactive install, assumes you agree with the Netop License.\n";

	while($arg = shift @ARGV) {
		if ($arg eq "--version") {
			print "$VERSION\n";
			exit;
		} 
		elsif ( $arg eq "--license" ) {
			print_license( 1 );
			exit ( 0 );
		} 
		elsif ($arg eq "--debug") {
			$DEBUG = 1;
		} 
		elsif ($arg eq "--autoinstall") {
        	$autoinstall_flag = "--autoinstall";
			$autoinstall = 1;
		} 
		elsif ($arg eq "--help") {
			print $usage;
			exit;
		} 
		elsif ($arg eq "--serial") {
			# next token is not another option ?
			if ( ( defined $ARGV[0] ) and ( "--" ne substr($ARGV[0],0,2) ) ) { 
				$host_serial = shift @ARGV;
			} 
				else {
					die "bad command line";
			} 
		} 
		else {
			print "Error: unknown arg: '$arg'\n";
			die $usage;
		}
	}

	die "Must have super-user privileges to install this application.\nPlease elevate your privileges with 'su -' and try again.\n"
		if(! su_test());

	system_setup();
	
	# determine if rpm-files available
        my $rpmfiles = `/bin/ls netop*.rpm 2>/dev/null`;
	$rpmfiles =~ s/\n//g;
	if($rpmfiles eq "") {
		die "RPM package not found\n";
	}

	my $install_dir = `pwd`;
	$install_dir =~ s/\n//g;

	# check if the configuration file already exists
	my $config_file_exists = 0;
	if (-e $HOST_INIFILE) {
		$config_file_exists = 1;
	}

        print "\nInstalling Netop Host $NETOP_PRODUCT_VERSION\n";
	my $result = 0;
	
	# verify if the current OS is supported, if not, give a warning
	my $supported = 1;
	if ($OS_ID =~ "sles") {
		if ($OS_VER gt $MAX_SUPPORTED_OS_VER || $OS_VER lt $MIN_SUPPORTED_OS_VER) {
			$supported = 0;
		}
	}
	else {
		$supported = 0;
	}
	if (!$supported) {
		print "Detected OS $SYSTEM $OS_VER, which is not officially supported by this package.\n";
		if (!$autoinstall) {
			my $answer = input_confirmation_Y("Do you still want to continue with the installation?");
			$answer = lc substr($answer,0,1);
			if ( 'y' ne $answer )
			{
				exit;
			}
		}
		else {
			print "Will attempt to continue because of --autoinstall flag.\n";
		}
	}
	
	# make sure rmp utility exists, otherwise installation cannot succeed
	if (! -e '/bin/rpm') {
		print STDERR "The RPM Package Manager (rpm command) has not been found on this system. \nThe installation cannot continue.\n";
		exit;
	}
	
	if ( $autoinstall ) {
		if ( $host_serial eq "" ) {
			print "Note: you did not provide any license (--serial <serial>). A trial license will be used.\n";
        }
	}
	
	my $deb_ca_cert = "/etc/ssl/certs/ca-certificates.crt";
	my $rh_ca_cert = "/etc/pki/tls/certs/ca-bundle.crt";
	my $sles_ca_cert = "/etc/ssl/ca-bundle.pem";

	my $default_ca_cert = "ca-certificates.crt";
	my $ca_cert_dir = "/var/opt/netop/certs/";
	my $ca_cert_file = "/var/opt/netop/certs/ca-bundle.crt";
	
	#make sure the ca certificate is available in certificates directory
	print "Installing CA certificates bundle in $ca_cert_dir\n";

	unless (-e $ca_cert_file) {
		eval { mkpath($ca_cert_dir) unless (-e $ca_cert_dir) };
		if ($@) {
			print "Couldn't create $ca_cert_dir: $@\n";
		}
		if (-e $deb_ca_cert) {
			symlink($deb_ca_cert, $ca_cert_file);
		}
		elsif (-e $rh_ca_cert) {
			symlink($rh_ca_cert, $ca_cert_file);
		}
		elsif (-e $sles_ca_cert) {
			symlink($sles_ca_cert, $ca_cert_file);
		}
		else {
			die "No CA certificate" unless (-e $default_ca_cert);
			copy($default_ca_cert,  $ca_cert_file);
		}    
	}

        my $rpmfile = `/bin/ls netop-host*.rpm`;
	$rpmfile =~ s/\n//g;
	die "package not found\n" if(!($rpmfile =~ m|.*$NETOP_PRODUCT_VERSION|));
	# query for already installed product (old package name format)
	$cmd = "/bin/rpm -q NetopHost";
	my $installed = `$cmd`;
	$installed =~ s/\n//g;
	if($installed =~ m{NetopHost-(\d{1,2})[\.|-](\d{2})-(........)}) {
		dbg("NetopHost allready installed : '$installed' !\n");
		if(-e "/etc/init.d/netophostd") {
			$cmd = "/etc/init.d/netophostd stop >/dev/null 2>/dev/null";
			`$cmd`;
		}
		# uninstall without running %preun, which deletes the configuration
		$cmd = "/bin/rpm -e --noscripts $installed";
		$res = system("$cmd");
		if($res != 0) {
			print STDERR "Install Host: '$cmd' failed with status $res, trying to upgrade.\n";
		}
	}
	# query for already installed product (new package name format)
	$cmd = "/bin/rpm -q netop-host";
	$installed = `$cmd`;
	$installed =~ s/\n//g;
	if($installed =~ m{netop-host-(\d{1,2}).(\d{2})}) {
		dbg("NetopHost allready installed : '$installed' !\n");
		if(-e "/etc/init.d/netophostd") {
			$cmd = "/etc/init.d/netophostd stop >/dev/null 2>/dev/null";
			`$cmd`;
		}
		if (-e '/usr/bin/yum') {
			$cmd = "/usr/bin/yum erase -y $installed";
		}
		else {
			$cmd = "/bin/rpm -e $installed";
		}
		`$cmd`;
	}

        system("./installpubkey '$install_dir/netop.pub'");
	my $command = "";
		
	if(-e '/usr/bin/zypper') {
		if ( $autoinstall ) {
			$command = "/usr/bin/zypper --non-interactive in $rpmfile";
		}
		else {
			$command = "/usr/bin/zypper in $rpmfile";
		}	
		my $res = system($command);
		if($res != 0) {
			print STDERR "install_rpm: '$command' failed with status $res\n";
			$result = 0;
		}
		else {
			$result = 1;
		}
	}
	else {
		$command = "/bin/rpm -U --nosignature $rpmfile";
		my $res = system($command) ;

		if($res != 0) {
			print STDERR "install_rpm: '$command' failed with status $res\n";
			$result = 0;
		}
		else {
			$result = 1;
		}
	}

	# installation OK :
	if ( 1 != $result ) {
		# 'hard' failure of installer :
		if ( 0 == $result ) {
			print STDERR "\nInstallation FAILED!.\n";
			exit( 1 );
		}
		# user chose N(o) in gdebi interactive install :
		if ( 2 == $result ) {
			print STDERR "You chose not to install this package on this system.\n";
			print STDERR "Installer can not continue for this product.\n";
			exit( 1 );
		}
	}

	convert_host_files();
		
	my $configure_script = "/opt/netop/host/configure/configure.pl";
	if (! -e $configure_script) {
                print "Netop Host installed but not yet configured for use\n";
		exit;
	}
		
	my $cmd = "perl $configure_script --serial $host_serial $autoinstall_flag";
	if ($config_file_exists == 1) {
		$cmd = $cmd . " --upgrade"
	}
	dbg("cmd: $cmd\n");
	my $st = system($cmd);
	$st >>= 8;
	dbg("configure.pl st: $st\n");
	if ($st == 0) { 
	} 
	else {
		if ($st == 56) {
                        $method = "rpm -e netop-host";
                        print "\t(run '$method' to uninstall Netop Host)\n"; # TODO
		}
	}
}

main;
exit;
__END__
