#!/usr/bin/perl -w
#
# configure Netop Host installed using rpm or tar.gz
# (c) Netop Business Solutions A/S 2016
#

$VERSION = "configure: 1.08";
$DEBUG = 0;
$NETOPHOST = "/usr/sbin/netophost";
$QXMLEDIT = "/usr/bin/qxmledit";
$NETOPHOSTXD = "/usr/sbin/netophostxd";
$INITD_NETOPHOSTD = "/etc/init.d/netophostd";

$UNCONFIGURE_SCRIPT = "/var/opt/netop/host/unconfigure.pl";
$UNCONFIGURE_COMMANDS = "";
$INIFILE = "/var/opt/netop/host/host.xml";

use Cwd;
use File::Basename;

my $CHKCONFIG_CMD = "/sbin/chkconfig";
my $CP_CMD    = "/bin/cp";
my $MV_CMD    = "/bin/mv";
my $LN_CMD    = "/bin/ln";
my $LS_CMD    = "/bin/ls";
my $FIND_CMD  = "/usr/bin/find";
my $CAT_CMD   = "/bin/cat";
my $ID_CMD    = "/usr/bin/id";
my $D2U_CMD   = "/usr/bin/dos2unix";
my $CHMOD_CMD = "/usr/bin/chmod";
my $SYSTEMCTL_CMD = "/bin/systemctl";

my $SCRIPTDIR=dirname(__FILE__);
my $CRTDIR=cwd();
my $THIS="$0";

my $USER_LICENSE_FILE = "license.dat";
my $LICENSE_DESTINATION_OLD = "/etc/NetopHost";
my $LICENSE_DESTINATION = "/var/opt/netop/host";
my $LOCAL_LICENSE_FILE = ".license.dat";

my $SYSTEM = "";
my $HOSTXD_WORKS = "no";
my $OS_VER = "";
my $OS_BIT = "";
my $QUERY_RES = "start_status:";

sub write_unconfigure
{
   open OUTFILE,">$UNCONFIGURE_SCRIPT" or die "Cannot open '$UNCONFIGURE_SCRIPT' for writing: $!\n";
   printf OUTFILE "%s",$UNCONFIGURE_COMMANDS;
   close OUTFILE;
}

sub system_setup
{
    my $uname = `uname`;
    dbg("uname  = $uname");
    if ($uname =~ "SunOS") {
          $SYSTEM = "solaris";
          $OS_VER = `uname -r`;
    } elsif ($uname =~ "Linux") {
          $SYSTEM = "Linux";
          my $release = "/etc/redhat-release";
          if (-e $release) {
                $SYSTEM = "RedHat";
                open FILE,"<$release";
                $line=<FILE>;
                if($line eq "Red Hat Enterprise Linux AS release 4 (Nahant)") {
                    $HOSTXD_WORKS="yes";
                }
          }
          $release = "/etc/SuSE-release";
          if (-e $release) {
                $SYSTEM = "SuSE";
          }
          $release = "/etc/depmod.d/ubuntu.conf";
          if (-e $release) {
                $SYSTEM = "ubuntu";
          }
    } else {
          dbg("Warning: cannot identify UNIX system.\n");
          return 0;
    }
    $OS_VER =~ s/\n//g;
    $OS_BIT = `uname -m`;
    dbg("SYSTEM = $SYSTEM\n");
    dbg("OS_VER = $OS_VER\n");
    return 1;
}

sub get_tool
{
    my ($arg) = @_;
    my $tool="";
    my $cmd = "unalias cp 2>/dev/null; which $arg 2>/dev/null";
    dbg("cmd: $cmd\n");
    $tool = `$cmd`;
    $tool =~ s/\n//g;
    if($tool =~ m|[ \t]|) {
       $tool = ""; # contained whitespace, must be a message.
    }

    return $tool;
}

sub setup_tools
{
    if(!(-e $CHKCONFIG_CMD)) {
        $CHKCONFIG_CMD =  get_tool("chkconfig");
    }
    
    if(!(-e $SYSTEMCTL_CMD)) {
        $SYSTEMCTL_CMD =  get_tool("systemctl");
    }
    
    dbg("setup_tools: CHKCONFIG_CMD = $CHKCONFIG_CMD\n");
    if(! -e $CP_CMD) {
        $CP_CMD = get_tool("cp");
        if(! -e $CP_CMD ) {
            die "Non-standard cp, please update CP_CMD\n";
        }
    }
    if(! -e $LN_CMD) {
        $LN_CMD = get_tool("ln");
        if(! -e $LN_CMD) {
            die "Non-standard ln, please update LN_CMD)\n";
        }
    }
    if(!-e $LS_CMD) {
        $LS_CMD = get_tool("ls");
        if(! -e $LS_CMD ) {
            die "Non-standard ls, please update LS_CMD\n";
        }
    }
    if(! -e $FIND_CMD) {
        $FIND_CMD = get_tool("find");
    }
    if(! -e $CAT_CMD) {
        $CAT_CMD = get_tool("cat");
        if(! -e $CAT_CMD) {
            die "Non-standard ln, please update CAT_CMD)\n";
        }
    }
    if(! -e $ID_CMD) {
        $ID_CMD = get_tool("id");
    }
    if(! -e $D2U_CMD) {
        $D2U_CMD = get_tool("dos2unix");
        if(! -e $D2U_CMD ) {
            $D2U_CMD = get_tool("fromdos");
            if(! -e $D2U_CMD) {
                die "Non-standard dos2unix, please update D2U_CMD\n";
            }
        }
    }
    if(! -e $CHMOD_CMD) {
        $CHMOD_CMD = get_tool("chmod");
        if(! -e $CHMOD_CMD ) {
            die "Non-standard chmod, please update CHMOD_CMD\n";
        }
    }
}

sub dbg
{
    if($DEBUG) {
    my $arg = shift(@_);
    print STDERR "DBG: $arg";
    }
}

sub input_confirmation
{
   my $options = "[yes/No/quit]";

   my $question = shift(@_);
   print "$question $options >";
   while(<STDIN>) {
       my $lower = lc $_;
       if( ($lower =~ ("y"))) {
           return "y";
       } elsif (($_ eq "\n") || ($lower =~ ("n"))) {
           return "n";
       } elsif (($lower =~ ("q"))) {
           return "q";
       } else {
		   print "Please reply: $options\n";
           print "$question $options >";
       }
   }
}

sub input_confirmation_N
{
   my $options = "[yes/No]";

   my $question = shift(@_);
   print "$question $options >";
   while(<STDIN>) {
		my $lower = lc $_;
		if( ($lower =~ ("y"))) {
			return "y";
		} 
		elsif (($_ eq "\n") || ($lower =~ ("n"))) {
			return "n";
		} 
        else {
			print "Please reply: $options\n";
            print "$question $options >";
       }
   }
}

sub input_confirmation_Y
{
   my $options = "[Yes/no]";

   my $question = shift(@_);
   print "$question $options >";
   while(<STDIN>) {
       my $lower = lc $_;
       if( ($lower =~ ("n"))) {
           return "n";
       } elsif (($_ eq "\n") || ($lower =~ ("y"))) {
           return "y";
       } else {
		   print "Please reply: $options\n";
           print "$question $options >";
       }
   }
}

#my $TRIAL_LICENSE=

#my $NML_LICENSE=

sub write_licfile
{
   my ($text,$licfile) = @_;
   open OUTFILE,">$licfile" or die "Cannot open '$licfile' for writing: $!\n";
   printf OUTFILE "%s",$text;
   close OUTFILE;
   print "License agreement written to $licfile\n\n";
}

# $_[0] is current license file (MANDATORY)
# $_[1] is destination folder   (MANDATORY)
# $_[2] in the new name of the license file to copy into destination (OPTIONAL) == basename($_[0])
# //
sub copy_license_to_installation_point
{
    ( $source, $destination, $new_name ) = @_;

    my $error = "\tInvalid arguments in : sub copy_license_to_installation_point.\n";
    goto return_error unless ( defined $source ) and ( defined $destination ); # sanity check
    $error = "\tInvalid license file : '$source' .\n";
    goto return_error unless ( (! -e $source ) or ( ! -f $source ) );
    
    $new_name = basename( $source ) unless defined $new_name;
    my $new_license = safe_license_to_target( $new_name, $destination );
    # copy the current license
    return $new_license unless ( 0 != system("$CP_CMD $source $destination_file") );
    $error = "\tFailed to copy '$source' to '$destination_file'!\n";
return_error:
    print STDERR "$error";
    return undef;
} # sub copy_license_to_installation_point

# generate the path target for new license file
# if there already is such a file, rename it to something like <old_file>.<[0-9]+>
# $_[0] is license file name
# $_[1] is target path
sub safe_license_to_target
{
    ( $license_file, $destination ) = @_;

    my $error = "\nInvalid arguments in : sub safe_license_to_target.\n";
    goto return_error unless ( defined $license_file ) and ( defined $destination ); # sanity check

    $error = "\tDestination '$destination' is invalid ( not a directory ) !\n";
    goto return_error unless ( -d $destination ); # destination is a directory ?

    $error = "\tDestination '$destination' is invalid ( not writable ) !\n";
    goto return_error unless ( -w $destination ); # destination is writable ?

    my $destination_file = $destination . '/' . $license_file;
    if ( -e $destination_file ) # already exists ?
    {
        $error =  "\tLocal license copy '$destination_file' already exists and it not a regular file !\n";
        goto return_error unless ! ( -f $destination_file );

        # make a backup copy with a random suffix :
        my $prefix_num = 0;
        my $previous_license_file = $destination_file;
        do
        {
            $previous_license_file = $destination_file . '.' .  $prefix_num;
        }
        while ( ! -e $previous_license_file );

        print "\tAn older version of '$destination_file' already exists, will save it to $previous_license_file.\n";
        return $destination_file unless ( 0 != system("$MV_CMD $destination_file $previous_license_file") );

        $error = "\tFailed to copy '$destination_file' to '$previous_license_file'!\n";
    } # if ( -e $destination_file )
    return $destination_file;
return_error:
    print STDERR "$error";
    return undef;
} # sub license_to_target

# $_[0] is a flag that tells wether to allow user to scroll or not ( OPTIONAL )
sub print_license
{
    my $scroll_method = "more";
    my $allow_scroll = shift;
    if ( ! defined $allow_scroll ) { $allow_scroll = 1; }
    else { $allow_scroll = 0 + $allow_scroll; } # force numeric ctx
    if ( ! defined $allow_scroll ) { $allow_scroll = 1; }
    my $command = "$CAT_CMD $SCRIPTDIR/eula.txt";
    if ( 1 == $allow_scroll ) { $command = $command . " | $scroll_method"; }
    system($command);
} # sub print_license  

sub host_validator_wrapper
{
    ( $serial, $license_path ) = @_;
    
    # (!!) binary that does the actual validation and license generation reads serial from ENVVAR (!!) wtf
    $ENV{"DW_INSTALL_SERIAL"} = $serial;
    return validate_host_serial_number( $license_path );
} # sub host_validator_wrapper

# $_[0] : a license file that user already has ( older license )
# $_[1] : name of the new license file
# $_[2] : destination of new license file
# $_[3] : serial number
# $_[4] : autoinstall (OPTIONAL) = 0 (no)
sub validate_with_license_or_serial
{
    ( $user_license, $new_license, $destination , $serial, $autoinstall ) = @_;

    $autoinstall = 0 unless defined $autoinstall;

    my $new_license_file = undef;
    # user has a serial number :
user_has_serial :
    if ( ( defined $serial ) and ( $serial ne "" ) )
    {
        print "\tWill validate Netop Host with provided license key.\n";
        # determine if there is a license file already ( from a previous install ) and make a copy :
        $new_license_file = safe_license_to_target( $new_license, $destination );
        return ( 0, $new_license_file ) unless defined $new_license_file;
        goto ask_for_serial unless host_validator_wrapper( $serial, $new_license_file );
        return ( 1, $new_license_file );
    }
    # user does not have a serial number :
    # but has a license file :
    elsif ( ( defined $user_license ) and ( -e $user_license ) )
    {
        print "\tFound previous license file : '$user_license' .\n";
        print "\t\tNetop Host will be validated with this licence.\n";
        # copy user's current license to installation directory :
        $new_license_file = copy_license_to_installation_point($user_license, $destination, $new_license);
        return ( 0, $new_license_file ) unless defined $new_license_file;
        return ( 1, $new_license_file ) unless host_validator_wrapper( $serial, $new_license_file );
        print "\n Your license '$user_license' could not be validated.\n";
    }
ask_for_serial:
    # doesn't have a license file
    # interactive install :
    $new_license_file = safe_license_to_target( $new_license, $destination ) unless defined $new_license_file;
    return ( 0, $new_license_file ) unless defined $new_license_file;
	if ( ! $autoinstall )
    {
        ( $validated, $serial ) = patiently_ask_for_serial("Netop Host", \&host_validator_wrapper, 3, $new_license_file);
        return ( 1, $new_license_file ) unless ! $validated;
        return ( 0, $new_license_file );
    }
	else 
	{
		return ( 1, $new_license_file ) unless host_validator_wrapper( "TRIAL", $new_license_file );
	}
    return ( 1 , $new_license_file );
} # sub validate_with_license_or_serial

sub host_installed_but_not_validated
{
    print "\nThe application has been installed, but none of the license keys could be validated.\n";
    print "Please run '$THIS' as super-user to configure and validate the product.\n";
    print "Thank you for using Netop Business Solutions A/S!\n\n";
} # sub host_installed_but_not_validated

sub host_installed_and_validated
{
    my $need_rc = shift;
    $need_rc = 0 unless ( defined $need_rc );
    print "\nNetop Host application has been installed and activated.\n";
	if ($need_rc) {
		print "Note: display manager needs to be restarted (or machine rebooted) for the changes to take full effect.\n";
	} else {
                print "You can start the Netop Host user interface by typing 'netophostgui' as a regular user. ";
		print "Please note that this only starts the GUI, the Host is already running as a different process.\n";
	}
    print "Thank you for using Netop Business Solutions A/S!\n\n";
} # sub host_installed_and_validated

# $_[0] is license_file
sub validate_host_serial_number
{
    my $license_file = shift;
    return 0 unless ( defined $license_file );
    # (!) actual serial number is passed as an environment variable (!)
    print "\tValidating license ...\n";
    my $result = 0;
    
    if ( (-e $license_file) || ($license_file eq "TRIAL") || ( $license_file eq "PORTAL" ) ) {
        $result = `$NETOPHOST --testlic --license.dat $license_file >/dev/null 2>/dev/null; echo \$?`;
    } else {
        $result = `$NETOPHOST --testlic  >/dev/null 2>/dev/null; echo \$?`;
    }
    
    $result =~ s/\s+$//;

    if ( 0 == $result )
    {
        unlink "$license_file"; # was in older implementation
        print "\t\tLicense key validated OK.\n";
        print "\t\tYour license has been saved.\n";
        
        return 1;
    }
    else
    {
        # from older implementation :
        if (($result > 27) && ($result != 101) && ($result != 102))
        {
            print STDERR "Unexpected error from validator $NETOPHOST : $result\nValidation failed!\nPlease contact technical support.\n\n";
            exit(1);
        }
        print "\t\tLicense Key is invalid.\n";
        return 0;
    }
} # sub validate_host_serial_number

# $_[0] is product name to display (MANDATORY)
# $_[1] is the validator subroutine passed as a reference (MANDATORY) that must return a boolean flag.
# $_[2] is the patience ( times to ask ) (OPTIONAL) = 3
# $_[3] is an optional argument to the validator (OPTIONAL) = undef   
sub patiently_ask_for_serial
{
    ( $product_name, $validator, $patience, $validator_arg ) = @_;
    $answer = undef;
    return ( 0, $answer ) unless ( ( defined $product_name ) and ( defined $validator ) );
    
    $patience = 3 unless ( ( defined $patience ) and ( ( $patience *= 1 ) ) ); # default patience : perl is patient

    while ( $patience )
    {
        my $options = "[T]rial / [p]ortal / [c]ustom";
        print "\n[$patience] Which type of license do you want to use for $product_name? $options: ";
        $answer = <STDIN>;
        
        my $lower = lc $answer;
        if ( ($lower eq "p\n"))
        { 
            $answer = "PORTAL"; 
        }
        elsif (($answer eq "\n") || ($lower eq "t\n")) 
        { 
            $answer = "TRIAL"; 
        }
        elsif (($lower eq "c\n")) 
        { 
            print "\n\t[$patience] Enter License key for $product_name ( or leave blank for trial ):\n";
            $answer = <STDIN>;

            if ( (!$answer) or ($answer =~ /^\s*$/) )  { $answer = "TRIAL"; } 
        }
        else
        {    
             print "Please reply: $options\n";
         next;  
        }      
        
        if ( $validator->( $answer, $validator_arg ) ) { return ( $patience, $answer ); }
        $patience--;
        if ($patience) {        
            print "\nThe license key could not be validated, please try again. ($patience)\n";
        }
    }
    
    return ( 0, $answer );
} # sub patiently_ask_for_serial

# $_[0] is the name of the input field to ask for (MANDATORY)
sub ask_for_password_input
{
    system('stty', '-echo');  # Disable echoing
    my $answer = ask_for_input(@_);
    system('stty', 'echo');   # Turn it back on
    print "\n";
    
    return $answer;
}

# $_[0] is the name of the input field to ask for (MANDATORY)
sub ask_for_input
{
    my $patience = 3;
    my $field_name = shift;            
    my $answer = undef;
    
    return undef unless defined ($field_name);
    
    while ( !defined($answer) && $patience--)
    {        
        print "\n$field_name : ";
        chomp($answer = <STDIN>);        
        
        #valid input and not empty and not all spaces
        if ( $answer && $answer ne "" && $answer !~ /^\s*$/) { 
            return $answer; 
        }      
        else { 
            $answer = undef;
        }
    }
    
    return undef;
} # sub ask_for_input

$SIGNAL_UNINSTALL=0;
sub check_license
{
    my ( $serial, $licensedto, $autoinstall ) = @_;
    print "\nThe following license-agreement apply to Netop Host:\n";
    print "[Please read carefully]\n";
    # interactive install
    my $answer = '';
    if ( 0 == $autoinstall )
    {
        print_license( ! $autoinstall );
        $answer = input_confirmation_N("Do you accept this license?");
        $answer = lc substr($answer,0,1);
        if ( 'y' ne $answer )
        {
            print "\tLicense not agreed upon, installer can not continue.\n";
            dbg("License not agreed.\n");
            $SIGNAL_UNINSTALL=1;
            return 0;
        }
    } 
    # non-interactive install
    # ( 1 == $autoinstall )
    else
    {
        print_license( ! $autoinstall );
        print "Auto-accepted license because of --autoinstall flag.\n";
    }
    if($licensedto ne "") { $ENV{"DW_INSTALL_LICENSED_TO"} = $licensedto; } # TODO : ??
	
	if (! -e $LICENSE_DESTINATION) {
		dbg("Folder $LICENSE_DESTINATION does not exist, creating now..\n");
		system("mkdir $LICENSE_DESTINATION");
	}

    ( $answer, $license_file ) = validate_with_license_or_serial($USER_LICENSE_FILE, $LOCAL_LICENSE_FILE, $LICENSE_DESTINATION, $serial, $autoinstall);
    # validated ok ?
    return 0 unless $answer;
    return 1;
} # sub check_license

sub has_portal_account
{
    system("$QXMLEDIT -f $INIFILE --test_portal_account");
    if ($? == 0) {
        return 1;
    }
    return 0;
}

sub init_portal_account
{
    my $is_upgrade = shift(@_);

    # if this is an upgrade from an older Host, do not prompt for Portal credentials
    if ($is_upgrade == 1) {
        return 1;
    }

    my $enrollment_key = undef;
    
    while(1) {
        my $answer = input_confirmation("Do you have a Netop Portal account?");
        $answer = lc substr($answer,0,1);        
        
        if ( $answer eq 'n') {           
            return 1;
        }

        if ( $answer eq 'q') {   
            return 0; #quit
        }
        
        if ( $answer ne 'y') {   
            next;
        }
        
        $enrollment_key = ask_for_input("Enrollment key");
        if (!$enrollment_key) { next; }

        last;
    }
        
    $enrollment_key = quotemeta($enrollment_key);
    
    system("$QXMLEDIT -f $INIFILE --portal_enrollment_key $enrollment_key");
    
    return 1;
}

sub install_chkconfig
{
    $cmd = "$CHKCONFIG_CMD --add netophostd >/dev/null";
    dbg("cmd = $cmd\n");
    my $rc = system($cmd);
    return 0 if($rc != 0);

    if(lc $SYSTEM eq "redhat") {
          $cmd = "$CHKCONFIG_CMD --level 3 netophostd on >/dev/null";
          dbg("cmd = $cmd\n");
          my $rc = system($cmd);
          return 0 if ($rc != 0);
          $cmd = "$CHKCONFIG_CMD --level 5 netophostd on >/dev/null";
          dbg("cmd = $cmd\n");
          $rc = system($cmd);
          return 0 if ($rc != 0);

          $UNCONFIGURE_COMMANDS .= "    \`$CHKCONFIG_CMD --del netophostd\`;\n";
          return 1;
    }
    if(lc $SYSTEM eq "ubun") {
          $cmd = "$CHKCONFIG_CMD --level 3 netophostd on >/dev/null";
          dbg("cmd = $cmd\n");
          my $rc = system($cmd);
          return 0 if ($rc != 0);
          $cmd = "$CHKCONFIG_CMD --level 5 netophostd on >/dev/null";
          dbg("cmd = $cmd\n");
          $rc = system($cmd);
          return 0 if ($rc != 0);

          $UNCONFIGURE_COMMANDS .= "    \`$CHKCONFIG_CMD --del netophostd\`;\n";
          return 1;
    }
    if(lc $SYSTEM eq "suse") {
          $cmd = "$CHKCONFIG_CMD netophostd 35 >/dev/null";
          dbg("cmd = $cmd\n");
          my $rc = system($cmd);
          return 0 if ($rc != 0);
          $UNCONFIGURE_COMMANDS .= "    \`$CHKCONFIG_CMD --del netophostd >/dev/null\`;\n";
          return 1;
    }
    return 0;
}

sub disable_chkconfig
{
    $cmd = "$CHKCONFIG_CMD netophostd off >/dev/null";
    dbg("cmd = $cmd\n");
    my $rc = system($cmd);
    return 0 if($rc != 0);
    return 1;
}

sub query_chkconfig
{
    my $cmd = "$CHKCONFIG_CMD --list netophostd 2>/dev/null";
    dbg("query_chkconfig: cmd: $cmd\n");
    my $output = `$cmd`;
    dbg("query_chkconfig: output: $output\n");
    if( lc $output =~ m|3[:]on|) {
    print "$QUERY_RES 1";
    return 1;
    } else {
    print "$QUERY_RES 0";
    return 1;
    }

    return 0;
}

sub link_script
{
    my ($service_dir)= @_;
    if( -e "$service_dir/S95netophostd") {
    unlink "$service_dir/S95netophostd";
    }
    if( -e "$service_dir/K05netophostd") {
    unlink "$service_dir/K05netophostd";
    }
    dbg("Creating service link scripts in $service_dir\n");
    my $cmd = "$LN_CMD -s $INITD_NETOPHOSTD $service_dir/S95netophostd";
    dbg("cmd = $cmd\n");
    system($cmd);
    $cmd = "$LN_CMD -s $INITD_NETOPHOSTD $service_dir/K05netophostd";
    dbg("cmd = $cmd\n");
    system($cmd);

    if( -e "$service_dir/S95netophostd") {
    $UNCONFIGURE_COMMANDS .= "unlink \"$service_dir/S95netophostd\";\n";
    }
    if( -e "$service_dir/K05netophostd") {
    $UNCONFIGURE_COMMANDS .= "unlink \"$service_dir/K05netophostd\";\n";
    }
    return 1;
}

sub install_links()
{
    my $service_dir = "/etc/rc2.d";
    if(-d $service_dir) {
    link_script($service_dir);
        if($SYSTEM eq "solaris") {
            return 1;
        }
    $service_dir =~ s/2/3/g;
    if( -d $service_dir) {
        link_script($service_dir);
    }
    $service_dir =~ s/3/5/g;
    if( -d $service_dir) {
        link_script($service_dir);
    }
    return 1;
    }

    $service_dir = "/etc/init.d/rc2.d";
    if(-d $service_dir) {
    link_script($service_dir);
    $service_dir =~ s/2/3/g;
    if( -d $service_dir) {
        link_script($service_dir);
    }
    $service_dir =~ s/3/5/g;
    if( -d $service_dir) {
        link_script($service_dir);
    }
    return 1;
    }

    $service_dir = "/etc/rc.d/rc2.d";
    if(-d $service_dir) {
    link_script($service_dir);
    $service_dir =~ s/2/3/g;
    if( -d $service_dir) {
        link_script($service_dir);
    }
    $service_dir =~ s/3/5/g;
    if( -d $service_dir) {
        link_script($service_dir);
    }
    return 1;
    } else {
        print "Warning: automatic configuration of runlevels  failed.\n";
        print "Please update relevant runlevel (2,3 or 5) manually.\n";
    return 0;
    }
}

sub disable_link_script
{
    my ($service_dir)= @_;
    unlink "$service_dir/S95netophostd";
    unlink "$service_dir/K05netophostd";
    return 1;
}

sub disable_links()
{
    my $service_dir = "/etc/rc3.d";
    if(-d $service_dir) {
    disable_link_script($service_dir);
    $service_dir =~ s/3/5/g;
    disable_link_script($service_dir);
    return 1;
    }

    $service_dir = "/etc/init.d/rc3.d";
    if(-d $service_dir) {
    disable_link_script($service_dir);
    $service_dir =~ s/3/5/g;
    disable_link_script($service_dir);
    return 1;
    }

    $service_dir = "/etc/rc.d/rc3.d";
    if(-d $service_dir) {
    disable_link_script($service_dir);
    $service_dir =~ s/3/5/g;
    disable_link_script($service_dir);
    return 1;
    } else {
    print "Warning: automatic configuration of runlevel 3 failed.\n";
    print "Please update runlevel 3 manually.\n";
    return 0;
    }
}

sub query_links()
{
    my $service_dir = "/etc/rc3.d";
    if(-d $service_dir) {
    if(-e "$service_dir/S95netophostd") {
        print "$QUERY_RES 1";
        return 1;
    } else {
        print "$QUERY_RES 0";
        return 1;
    }
    return 1;
    }

    $service_dir = "/etc/init.d/rc3.d";
    if(-d $service_dir) {
    if(-e "$service_dir/S95netophostd") {
        print "$QUERY_RES 1";
        return 1;
    } else {
        print "$QUERY_RES 0";
        return 1;
    }
    }

    $service_dir = "/etc/rc.d/rc3.d";
    if(-d $service_dir) {
    if(-e "$service_dir/S95netophostd") {
        print "$QUERY_RES 1";
        return 1;
    } else {
        print "$QUERY_RES 0";
        return 1;
    }
    return 1;
    }

    print "$QUERY_RES 0";
    return 0;
}

sub copy_startup()
{
    # cp platform specific start script to /etc/init.d
    unlink $INITD_NETOPHOSTD;

    if(lc $SYSTEM eq "redhat") {
          $cmd = "$D2U_CMD -n /opt/netop/host/init.d/initd.rh $INITD_NETOPHOSTD";
    } elsif (lc $SYSTEM eq "suse") {
          $cmd = "$D2U_CMD -n /opt/netop/host/init.d/initd.suse $INITD_NETOPHOSTD";
    } elsif (lc $SYSTEM eq "ubuntu") {
          $cmd = "$CAT_CMD /opt/netop/host/init.d/initd.ubuntu | $D2U_CMD > $INITD_NETOPHOSTD";
    }else {
          # includes "debian"
          $cmd = "$CAT_CMD /opt/netop/host/init.d/initd.gen | $D2U_CMD > $INITD_NETOPHOSTD";
    }

    dbg("cmd = $cmd\n");
    system($cmd);
    system("$CHMOD_CMD +x $INITD_NETOPHOSTD");
    if( ! -f $INITD_NETOPHOSTD) {
          die "Cannot create $INITD_NETOPHOSTD\n";
    }
    return 1;
}

sub enable_startup
{
    my $cmd="";
    dbg("Enable startup script in /etc/init.d !\n");

    my ($output_query_res)= @_;
    # enable startup at runlevel 3 and 5
    
    
    if($CHKCONFIG_CMD ne "") {
        if(!install_chkconfig()) {
            goto install_links;
        }
        $UNCONFIGURE_COMMANDS .= "unlink \"$INITD_NETOPHOSTD\";\n";
        if($output_query_res != 0) {
            print "$QUERY_RES 1";
        }
        return 1;
    }

install_links:
    install_links();
    
    if($SYSTEMCTL_CMD ne "") {
        dbg("systemctl = $SYSTEMCTL_CMD\n");
        system("$SYSTEMCTL_CMD daemon-reload");
    }
    
    $UNCONFIGURE_COMMANDS .= "unlink \"$INITD_NETOPHOSTD\";\n";
    if($output_query_res != 0) {
    print "$QUERY_RES 1";
    }

    return 1;
}

sub disable_startup()
{
    my $cmd="";
    dbg("Disable startup script in /etc/init.d !\n");

    # disable startup at runlevel 3 and 5
    if($CHKCONFIG_CMD ne "") {
    if(!disable_chkconfig()) {
        goto disable_links;
    }
    print "$QUERY_RES 0";
    return 1;
    }

disable_links:
    disable_links();
    print "$QUERY_RES 0";

    return 1;
}

sub query_startup()
{
    my $cmd="";

    # disable startup at runlevel 3 and 5
    if($CHKCONFIG_CMD ne "") {
    if(!query_chkconfig()) {
        goto query_links;
    }
    return 1;
    }

query_links:
    query_links();

    return 1;
}
# NetopHostXd begin
#
# locate kdm config file
#
sub kdm_config_file
{
     my @trydir_list = ("/etc","/opt");
    my $trydir="";
    my @kdmrc_list=("/etc/kde/kdm/kdmrc", 
                    "/etc/X11/xdm/kdmrc",
            "/etc/opt/kde3/share/config/kdm/Xservers"
            );
    my $kdmrc="";
    my $cmd="";
    my $xservers="";

    while($kdmrc = shift @kdmrc_list) {
          $kdmrc =~ s/\n//g;
          if( ! -e $kdmrc) {
                goto next_kdmrc;
          }
          dbg("From list, testing kdmrc: $kdmrc\n");
          $cmd = "$CAT_CMD $kdmrc";
          dbg("cmd: $cmd\n");
          my $file=`$cmd`;
          if($file =~ m|Xservers[ \t]*=[ \t]*(.+)|) {
                $xservers = $1;
                dbg("Possible Xservers: $xservers\n");
                if( -e $xservers) {
                     return $xservers;
                }
          }
        next_kdmrc:;
    }

    if($FIND_CMD eq "") {
        goto use_list;
    }
    while($trydir = shift @trydir_list) {
          dbg("trydir: $trydir\n");
          if(! -d $trydir) {
                goto next;
          }
          $cmd = "$FIND_CMD $trydir 2>/dev/null | grep kdmrc";
          dbg("cmd: $cmd\n");
          @kdmrc_list = `$cmd`;
          while($kdmrc = shift @kdmrc_list) {
                $kdmrc =~ s/\n//g;
                dbg("Possible kdmrc: $kdmrc\n");
                $cmd = "$CAT_CMD $kdmrc";
                dbg("cmd: $cmd\n");
                my $file=`$cmd`;
                if($file =~ m|Xservers[ \t]*=[ \t]*(.+)|) {
                     $xservers = $1;
                     dbg("Possible Xservers: $xservers\n");
                     if( -e $xservers) {
                          return $xservers;
                     }
                }
          }
        next:;
    }
  use_list:;
    my @configdirs=("/etc/X11/xdm",
                          "/usr/X11R6/lib/X11/xdm",
                          "/etc/kde2/kdm",
                          "/etc/opt/kde3/share/config/kdm",
                          "/opt/kde/share/config/kdm"
                          );
    while(my $xdir = shift @configdirs) {
          $xservers = "$xdir/Xservers";
          dbg("try from list: $xservers\n");
          if( -e $xservers) {
                return $xservers;
          }
    }

    return "";
}

#
# locate gdm config file
#
sub gdm_config_file
{
    my $try="/etc/X11/gdm/gdm.conf";
    if(-e $try) {
          return $try;
    }
    $try="/etc/gdm/custom.conf";
    if(-e $try) {
                  return $try;
    }
    return "";
}

#
# locate dtlogin config file
#
sub dtlogin_config_file
{
    my $try="/usr/dt/config/Xservers";
    if(-e $try) {
          return $try;
    }
    return "";
}


# 0 = not ok
# 1 = X-server configured
# 2 = NetopHostXd already configured
sub check_Xservers_file
{
    my $status = 0;
    my ($file) = @_;
    dbg("check_Xservers_file: $file\n");
    if (! -e $file) {
          return 0;
    }
    # line by line parse file and check the existance of
    # a ":0 ..... Xserver ... " line
    open INFILE,"<$file" || return 0;
    while ($line = <INFILE>) {
    if ($line =~ m|^[ \t]*:0(.*)|) {
        dbg("possible $line");
        # check if already configured.
        my $right = $1;
        if ($right =~ m|(.*)[ \t]+/usr/sbin/netophostxd\s*$|m) {
        $status = 2;
        goto closedown;
        } 
        elsif ($right =~ m|(.*)[ \t]+/usr/sbin/netophostxd[ \t]+(.*)$|m) {
        $status = 2;
        goto closedown;
        } 
        elsif ($right =~ m|(.*)[ \t]+/usr/X11R6/bin/X\s*$|m) {
        $status = 1;
        goto closedown;
        } 
        elsif ($right =~ m|(.*)[ \t]+/usr/X11R6/bin/X[ \t]+(.*)$|m) {
        $status = 1;
        goto closedown;
        } 
        elsif ($right =~ m|(.*)[ \t]+/usr/openwin/bin/Xsun\s*$|m) {
        $status = 1;
        goto closedown;
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/openwin/bin/Xsun[ \t]+(.*)$|m) {
        $status = 1;
        goto closedown;
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/X11/bin/Xserver\s*$|m) {
        $status = 1;
        goto closedown;
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/X11/bin/Xserver[ \t]+(.*)$|m) {
        $status = 1;
        goto closedown;
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/bin/X\s*$|m) {
        $status = 1;
        goto closedown;
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/bin/X[ \t]+(.*)$|m) {
        $status = 1;
        goto closedown;
        }
    }
    }
  closedown:
    close INFILE;
     dbg("check_Xservers_file: returning $status\n");
    return $status;
}

# 0 = not ok
# 1 = X-server configured    (wrong)
# 2 = NetOpHostXd configured (ok)
sub install_Xservers_file
{
    my $status = 0;
    my ($file) = @_;
    dbg("install_Xservers_file: $file\n");
    if(! -e $file) {
          return 0;
    }
    # backup
    if( -e "$file.netop") {
          unlink "$file.netop";
    }
    my $cmd = "$CAT_CMD < $file > $file.netop";
    dbg("cmd = $cmd\n");
    my $st = system($cmd);
    $st >>= 8;
    if($st != 0) {
          dbg("backup file failed\n");
          return 0;
    }
    # line by line parse file and check the existance of
    # a ":0 ..... Xserver ... " line
    my $infile = "$file.netop";
    open INFILE,"<$infile" || die "cannot open";
    if($? != 0) {
          dbg("install_Xservers_file: open INFILE failed ($?): $infile\n");
          system("ls -l $infile");
          return 0;
    }
    my $outfile = "$file";
    open OUTFILE,">$outfile";
    if($? != 0) {
          dbg("open OUTFILE failed ($?): $outfile\n");
          close INFILE;
          return 0;
    }
    while($line = <INFILE>) {
    if($line =~ m|^[ \t]*:0(.*)|) {
        my $right = $1;
        if ($right =~ m|(.*)[ \t]+/usr/sbin/netophostxd\s*$|m) {
        print OUTFILE $line;
        } 
        elsif ($right =~ m|(.*)[ \t]+/usr/sbin/netophostxd[ \t]+(.*)$|m) {
        print OUTFILE $line;
        } 
        elsif ($right =~ m|(.*)[ \t]+/usr/X11R6/bin/X\s*$|m) {
        print OUTFILE ":0$1 /usr/sbin/netophostxd\n";
        } 
        elsif ($right =~ m|(.*)[ \t]+/usr/X11R6/bin/X[ \t]+(.*)$|m) {
        print OUTFILE ":0$1 /usr/sbin/netophostxd $2\n";
        } 
        elsif ($right =~ m|(.*)[ \t]+/usr/openwin/bin/Xsun\s*$|m) {
        print OUTFILE ":0$1 /usr/sbin/netophostxd\n";
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/openwin/bin/Xsun[ \t]+(.*)$|m) {
        print OUTFILE ":0$1 /usr/sbin/netophostxd $2\n";
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/X11/bin/Xserver\s*$|m) {
        print OUTFILE ":0$1 /usr/sbin/netophostxd\n";
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/X11/bin/Xserver[ \t]+(.*)$|m) {
        print OUTFILE ":0$1 /usr/sbin/netophostxd $2\n";
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/bin/X\s*$|m) {
        print OUTFILE ":0$1 /usr/sbin/netophostxd\n";
        }
        elsif ($right =~ m|(.*)[ \t]+/usr/bin/X[ \t]+(.*)$|m) {
        print OUTFILE ":0$1 /usr/sbin/netophostxd $2\n";
        }
    } else {
        print OUTFILE $line;
    }
    }
  closedown:
    close INFILE;
    close OUTFILE;

    # check the produced file
    $status = check_Xservers_file($file);
    if ($status != 2) {
          # hm. not updated, restore previous
          dbg("check_Xservers_file: failed, restoring backup\n");
          $cmd = "$CAT_CMD < $file.netop > $file";
          dbg("cmd: $cmd\n");
          system("$cmd");
    }
    if( -e "$file.netop") {
          unlink "$file.netop";
    }
    return $status;
}


sub real_Xserver
{
    my $Xpath = `which X`;
    chomp $Xpath;

    my @list = ("/usr/X11R6/bin/X", "/usr/openwin/bin/Xsun", "/usr/X11/bin/Xserver", "$Xpath");
    while (my $server = shift(@list)) {
#   print "MLU server: $server\n";
    if (-e $server) {
        return $server;
    }
    }
    return "";
}


# 0 = not ok
# 1 = X-server configured    (ok)
# 2 = NetOpHostXd configured (wrong)
sub uninstall_Xservers_file
{
    my $status = 0;
    my ($file) = @_;

    dbg("uninstall_Xservers_file: $file\n");
    if(! -e $file) {
          return 0;
    }
    my $real_xserver = real_Xserver();
    dbg("real_xserver = $real_xserver\n");
    if($real_xserver eq "") {
          return 0;
    }
    # backup
    if( -e "$file.netop") {
          unlink "$file.netop";
    }
    my $cmd = "$CAT_CMD < $file > $file.netop";
    dbg("cmd = $cmd\n");
    my $st = system($cmd);
    $st >>= 8;
    if($st != 0) {
          dbg("backup file failed\n");
          return 0;
    }

    # line by line parse file and check the existance of
    # a ":0 ..... Xserver ... " line
    my $infile = "$file.netop";
    open INFILE,"<$infile";
    if($? != 0) {
          dbg("uninstall_Xservers_file: open INFILE failed ($?): $infile\n");
          system("ls -l $infile");
          return 0;
    }

    my $outfile = "$file";
    open OUTFILE,">$outfile";
    if($? != 0) {
          dbg("open OUTFILE failed ($?): $outfile\n");
          close INFILE;
          return 0;
    }
    while ($line = <INFILE>) {
          if ($line =~ m|^[ \t]*:0(.*)|) {
                my $right = $1;
                if ($right =~ m|(.*)[ \t]+/usr/sbin/netophostxd\s*$|m) {
                     dbg("uninstall_Xservers_file::0$1 $real_xserver\n");
                     print OUTFILE ":0$1 $real_xserver\n";
                }
                elsif ($right =~ m|(.*)[ \t]+/usr/sbin/netophostxd[ \t]+(.*)$|m) {
                     dbg("uninstall_Xservers_file::0$1 $real_xserver $2\n");
                     print OUTFILE ":0$1 $real_xserver $2\n";
                }
          } else {
                print OUTFILE $line;
          }
    }
    closedown:
    close INFILE;
    close OUTFILE;

    # check the produced file
    $status = check_Xservers_file($file);
    if ($status != 1) {
          # hm. not updated, restore previous
          dbg("check_Xservers_file: failed, restoring backup\n");
          $cmd = "$CAT_CMD < $file.netop > $file";
          dbg("cmd: $cmd\n");
          system("$cmd");
    }
    if( -e "$file.netop") {
          unlink "$file.netop";
    }
    return $status;
}

# 0 = not ok
# 1 = X-server configured
# 2 = NetOpHostXd configured
sub check_gdm_file
{
    my $state  = 0;
    my $status = 0;
    my ($file) = @_;
    dbg("check_gdm_file: $file\n");
    if (! -e $file) {
        return 0;
    }
    open INFILE,"<$file" || return 0;
    while (my $line = <INFILE>) {
        if ($state == 0) {
            # locate section [servers]
            $line = lc $line;
            if ($line =~ m|^\[servers\]|) {
                dbg("possible section $line");
                $state = 1;
            }
        } elsif ($state == 1) {
            # locate "0=server"
            if($line =~ m|^[ \t]*0=[ \t]*(.*)|) {
                my $server = $1;
                dbg("possible server '$server'\n");
                if ($server eq "/usr/sbin/netophostxd") {
                    $status = 2;
                    goto closedown;
                }
                $status = 1;
                goto closedown;
            }
        }
    }
    closedown:;
    close INFILE;
    return $status;
}

# 0 = not ok
# 1 = X-server configured
# 2 = NetOpHostXd configured
sub install_gdm_file
{
    my $state  = 0;
    my $status = 0;
    my ($file) = @_;
    dbg("check_gdm_file: $file\n");
    if(! -e $file) {
        return 0;
    }
    # backup
    if( -e "$file.netop") {
        unlink "$file.netop";
    }
    my $cmd = "$CAT_CMD < $file > $file.netop";
    dbg("cmd = $cmd\n");
    my $st = system($cmd);
    $st >>= 8;
    if($st != 0) {
        dbg("backup file failed\n");
        return 0;
    }
    my $infile = "$file.netop";
    open INFILE,"<$infile" || die "cannot open $infile";
    if($? != 0) {
        dbg("install_Xservers_file: open INFILE failed ($?): $infile\n");
        system("ls -l $infile");
        return 0;
    }
    my $outfile = "$file";
    open OUTFILE,">$outfile";
    if($? != 0) {
        dbg("open OUTFILE failed ($?): $outfile\n");
        close INFILE;
        return 0;
    }
    while(my $line = <INFILE>) {
        if($state == 0) {
            # locate section [servers]
            $tst_line = lc $line;
            if($tst_line =~ m|^\[servers\]|) {
                dbg("possible section $line");
                $state = 1;
            }
            print OUTFILE $line;
        } elsif ($state == 1) {
            # locate "0=server"
            if($line =~ m|^[ \t]*0=[ \t]*(.*)|) {
                my $server = $1;
                dbg("possible server '$server'\n");
                if($server eq $NETOPHOSTXD) {
                    print OUTFILE $line;
                } else {
                    # backup existing line
                    print OUTFILE "#netop_save $line";
                    # TBD: options
                    print OUTFILE "0=$NETOPHOSTXD\n";
                }
                $state = 2;
            }
        } else {
            print OUTFILE $line;
        }
    }
    closedown:;
    close INFILE;
    close OUTFILE;

    # check the produced file
    $status = check_gdm_file($file);
    if($status != 2) {
         # hm. not updated, restore previous
         dbg("check_gdm_file: failed, restoring backup\n");
         $cmd = "$CAT_CMD < $file.netop > $file";
         dbg("cmd: $cmd\n");
         system("$cmd");
    }
    if( -e "$file.netop") {
        unlink "$file.netop";
    }
    return $status;
}

# 0 = not ok
# 1 = X-server configured
# 2 = NetOpHostXd configured
sub uninstall_gdm_file
{
    my $state  = 0;
    my $status = 0;
    my ($file) = @_;
    dbg("check_gdm_file: $file\n");
    if(! -e $file) {
        return 0;
    }
    # backup
    if( -e "$file.netop") {
        unlink "$file.netop";
    }
    my $cmd = "$CAT_CMD < $file > $file.netop";
    dbg("cmd = $cmd\n");
    my $st = system($cmd);
    $st >>= 8;
    if($st != 0) {
        dbg("backup file failed\n");
        return 0;
    }
    my $infile = "$file.netop";
    open INFILE,"<$infile" || die "cannot open $infile";
    if($? != 0) {
        dbg("install_Xservers_file: open INFILE failed ($?): $infile\n");
        system("ls -l $infile");
        return 0;
    }
    my $outfile = "$file";
    open OUTFILE,">$outfile";
    if($? != 0) {
        dbg("open OUTFILE failed ($?): $outfile\n");
        close INFILE;
        return 0;
    }
    while(my $line = <INFILE>) {
        if($state == 0) {
            # locate section [servers]
            $tst_line = lc $line;
            if($tst_line =~ m|^\[servers\]|) {
                dbg("possible section $line");
                $state = 1;
            }
            print OUTFILE $line;
        } elsif ($state == 1) {
            # locate #netop_save or 0=
            # locate "0=server"

            if($line =~ m|^#netop_save (.*)|) {
                $state = 3;
                $server = $1;
                dbg("restored netop_save line: '$server'");
                print OUTFILE "$server\n";
            } elsif ($line =~ m|^[ \t]*0=[ \t]*(.*)|) {
                $server = $1;
                dbg("found backed up server line: $server\n");
                if($server =~ $NETOPHOSTXD) {
                    # skip line
                } else {
                    print OUTFILE $line;
                }
                $state = 2;
            }
        } elsif ($state == 3) {
            if ($line =~ m|^[ \t]*0=[ \t]*(.*)|) {
                # skip line, netop_save line already inserted
                $state = 4;
            } else {
                print OUTFILE $line;
            }
        } else {
            print OUTFILE $line;
        }
    }
    closedown:;
    close INFILE;
    close OUTFILE;

    # check the produced file
    $status = check_gdm_file($file);
    if($status != 1) {
         # hm. not updated, restore previous
         dbg("check_gdm_file: failed, restoring backup\n");
         $cmd = "$CAT_CMD < $file.netop > $file";
         dbg("cmd: $cmd\n");
         system("$cmd");
    }
    if( -e "$file.netop") {
        unlink "$file.netop";
    }
    return $status;
}

my $NEED_RESTART=0;

sub install_netophostxd
{
     my $ret_status=1;
    my $server = $NETOPHOSTXD;
    if (! -e $server) {
        print "install_netophostxd: server not found: $server\n";
        return 0;
    }
    my $Xservers = "";
    if ($SYSTEM eq "solaris") {
          $Xservers = dtlogin_config_file();
    } else {
          $Xservers = kdm_config_file();
    }
    
    print "xserver kdm: $Xservers\n";
    if ($Xservers eq "") {
          goto gdm_install;
    }
    my $status = check_Xservers_file($Xservers);
    if ($status == 0) {
    dbg("check_Xservers_file error\n");
        $ret_status = 0;
          goto gdm_install;
    }
    if ($status == 2) {
          # already installed.
          goto gdm_install;
    }
    $status = install_Xservers_file($Xservers);
    $NEED_RESTART = 1;
    if ($status == 0) {
    dbg("install_Xservers_file error\n");
        $ret_status = 0;
          goto gdm_install;
    }
    if ($status != 2) {
        print "install_netophostxd: problems installing netophostxd in $Xservers\n";
        $ret_status = 0;
          goto gdm_install;
    }
     
  gdm_install:;
    my $gdmconf = gdm_config_file();
    if ($gdmconf eq "") {
        goto fino;
    }
    $status = check_gdm_file($gdmconf);
    if ($status == 0) {
    dbg("check_gdm_file error\n");
        $ret_status = 0;
        goto fino;
    }
    if ($status == 2) {
          # already installed.
        goto fino;
    }
    $status = install_gdm_file($gdmconf);
    $NEED_RESTART = 1;
    if ($status == 0) {
    dbg("install_gdm_file error\n");
        $ret_status = 0;
        goto fino;
    }
    if ($status != 2) {
          print "install_netophostxd: problems installing netophostxd in $gdmconf\n";
        $ret_status = 0;
        goto fino;
    }
  fino:;
    return $ret_status;
}

sub uninstall_netophostxd
{   my $ret_status=1;
    my $server = $NETOPHOSTXD;
    if (! -e $server) {
        print "uninstall_netophostxd: server not found: $server\n";
        return 0;
    }
    my $Xservers = "";
    if ($SYSTEM eq "solaris") {
          $Xservers = dtlogin_config_file();
    } else {
          $Xservers = kdm_config_file();
    }
    if($Xservers eq "") {
        goto gdm_uninstall;
    }
    my $status = check_Xservers_file($Xservers);
    if ($status == 0) {
        $ret_status = 0;
        goto gdm_uninstall;
    }
    if ($status == 1) {
        # already uninstalled.
        goto gdm_uninstall;
    }
    $status = uninstall_Xservers_file($Xservers);
    $NEED_RESTART = 1;
    if($status == 0) {
        $ret_status = 0;
        goto gdm_uninstall;
    }
    if ($status != 1) {
          print "uninstall_netophostxd: problems uninstalling netophostxd in $Xservers\n";
          $ret_status = 0;
          goto gdm_uninstall;
    }

    gdm_uninstall:;
    my $gdmconf = gdm_config_file();
    if($gdmconf eq "") {
        goto fino;
    }
    $status = check_gdm_file($gdmconf);
    if ($status == 0) {
        $ret_status = 0;
        goto fino;
    }
    if ($status == 1) {
        # already uninstalled.
        goto fino;
    }
    $status = uninstall_gdm_file($gdmconf);
    $NEED_RESTART = 1;
    if ($status == 0) {
        $ret_status = 0;
        goto fino;
    }
    if ($status != 1) {
          print "uninstall_netophostxd: problems uninstalling netophostxd in $gdmconf\n";
          $ret_status = 0;
          goto fino;
    }
    fino:;
    return $ret_status;
}
# NetOpHOstXd end



#
# solaris crle config begin
#
my $CRLE_CONFIG  = "/var/ld/ld.config";
my $LDD_BINARY = "/bin/ldd";
my $CRLE_PATH = "";

sub configure_crle()
{
    return 1 if($CRLE_PATH eq "");

    if(-f $CRLE_CONFIG) {
          # file exist, use update flag: -u
          $cmd = "crle -u -l $CRLE_PATH";
    } else {
          # file does not exist, remember to include /usr/lib
          $cmd = "crle -l /usr/lib:$CRLE_PATH";
    }
    dbg("configure_crle: crle_command: '$cmd'\n");
    if( system $cmd ) {
          die "error in $cmd\n";
    }
}

#
# BEGIN ldd test tools
#
$FILE_EXPR  = "";
$SEARCH_DIR = "";
# locate $FILE_EXPR in $SEARCH_DIR
# return 1 if match, 0 if no match
sub find_file()
{
    my @filelist = `/bin/ls $SEARCH_DIR`;
    while ($file = pop @filelist) {
       return 1 if($file =~ m|$FILE_EXPR| );
    }
    return 0;
}

sub append_crle_path()
{
    if($CRLE_PATH eq "") {
          $CRLE_PATH = $SEARCH_DIR;
    } else {
          if(!($CRLE_PATH =~ $SEARCH_DIR)) {
                $CRLE_PATH = "$CRLE_PATH:$SEARCH_DIR";
          }
    }
}

$SO_MISSING_LIST = "";
$MISSING = "";
sub append_so_missing_list()
{
    if($SO_MISSING_LIST eq "") {
    $SO_MISSING_LIST = $MISSING;
    } else {
    if(!($SO_MISSING_LIST =~ $MISSING)) {
        $SO_MISSING_LIST = "$SO_MISSING_LIST, $MISSING";
    }
    }
}

# build directory list for CRLE -> $CRLE_PATH.
# return 0 if all ok.
# return 1 if CRLE path needs updated $CRLE_PATH is updated.
# return 2 if unresolved libraries exist (needs to be installed on system).
sub test_ldd
{
    my $st=0;
    my ($exe) = @_;
    my @ld = `$LDD_BINARY $exe`;
    while ($line = pop @ld) {
       my $WSP = "[ \t]";
       my $NWSP = "[^ \t]";
       my $regexp = "^$WSP*(lib$NWSP+.so$NWSP*)$WSP+=>$WSP+[(][^)]+[)].*\$";
       if($line =~ m|$regexp| ) {
       $missing = $1;
       if($missing =~ m|libXtst.so| ) {
           $SEARCH_DIR = "/usr/openwin/lib";
           $FILE_EXPR  = "libXtst.so";
           if(find_file() == 1) {
           if($VERBOSE) {print "Found unresolved library '$missing' in '$SEARCH_DIR'\n"};
           $st = 1 if($st < 1);
           append_crle_path();
           } else {
           if($VERBOSE) {print "Missing library: '$missing' pls. install\n"};
           $MISSING = $missing;
           append_so_missing_list();
           $st = 2 if($st < 2);
           }
       } elsif ($missing =~ m|libstdc[+][+].so| ) {
           $SEARCH_DIR = "/usr/local/lib";
           $FILE_EXPR  = "libstdc[+][+].so";
           if(find_file() == 1) {
           if($VERBOSE) {print "Found unresolved library '$missing' in '$SEARCH_DIR'\n"};
           $st = 1 if($st < 1);
           append_crle_path();
           } else {
           if($VERBOSE) {print "Missing library: '$missing' pls. install\n"};
           $MISSING = $missing;
           append_so_missing_list();
           $st = 2 if($st < 2);
           }
       } elsif ($missing =~ m|libgcc| ) {
           $SEARCH_DIR = "/usr/local/lib";
           $FILE_EXPR  = "libgcc";
           if(find_file() == 1) {
           if($VERBOSE) {print "Found unresolved library '$missing' in '$SEARCH_DIR'\n"};
           $st = 1 if($st < 1);
           append_crle_path();
           } else {
           if($VERBOSE) {print "Missing library: '$missing' pls. install\n"};
           $MISSING = $missing;
           append_so_missing_list();
           $st = 2 if($st < 2);
           }
       } else {
           if($VERBOSE) {printf "Missing ref to $1 cannot resolve.\n"};
           append_so_missing_list();
           $MISSING = $missing;
           $st = 2 if($st < 2);
       }
       }
    }
    return $st;
}
#
# END ldd test tools
#

#
# Forall executable: ensure that it can be run.
#
sub verify_requirements
{
    my $fatal_missing=0;
    my @exe_filelist = @_;
    while(my $exe = pop @exe_filelist) {
        die "Missing executable: '$exe'\n" if(! -e $exe);
        my $st = test_ldd($exe);
    if($st > $fatal_missing)  { $fatal_missing = 1};
    if($st > $fatal_missing)  { $fatal_missing = 2};
    }

    return $fatal_missing;
}

my $NEED_LD_PATH=0;
sub ensure_solaris_environment
{
    if($SYSTEM ne "solaris") {
        return 1;
    }
    my $ld_library_path = $ENV{"LD_LIBRARY_PATH"};
    my $st = verify_requirements("/usr/sbin/netophost");
    dbg("CRLE_PATH = '$CRLE_PATH'\n");
    # append LD_LIBRARY_PATH to crle if exist.
    if($st == 2) {
         die "Missing dynamic libraries on system: $SO_MISSING_LIST";
    }
    if($st != 0 || ($CRLE_PATH ne "")) {
        configure_crle();
    }
    if(verify_requirements("/usr/sbin/netophost")) {
        my $VER="8";
        if($OS_VER eq "5.9") {
            $VER = "9";
        }
        if(! -d "/usr/local/lib" || ! -e "/usr/local/lib/libstdc++.so.5") {
            print "! \n";
            print "! Warning: required library: libstdc++.so.5 is not installed on the system.\n";
            print "! libstdc++.so.5 is located in package SMClibgcc.\n";
            print "! SMClibgcc can be downloaded from www.sunfreeware.com\n";
            print "! or from any of the mirror sites.\n";
            print "! (Download the file: libgcc-3.2.X-sol$VER-sparc-local)\n";
            print "! \n";
            print "! Install using pgkadd:\n";
            print "! pkgadd -d libgcc-3.2-sol$VER-sparc-local\n";
            print "! \n";
            return 0;
        }
        print "! Warning:\n"
             ."! Cannot configure Runtime Environment (crle), please configure manually.\n"
             ."! info: 'ldd /usr/sbin/netophost' shows unresolved libraries.\n"
             ."! end Warning.\n";
        $NEED_LD_PATH=1;
        return 0;
    }
    return 1;
}
#
# solaris crle config end
#

sub su_test
{
    if(! -e $ID_CMD) {
        return 1; # continue, cannot get uid.
    }
    my $id = `$ID_CMD`;
    if($id =~ m|uid=([0-9]+)|) {
        if($1 eq "0") {
            return 1;
        }
    }
    return 0;
}

sub setup_pam()
{
# 	if ($OS_BIT =~ "i686"){
# 		return;
#     }
    my $prefix = "";
    if(-e "/lib32/security/pam_unix.so"){
        $prefix = "/lib32/security/";
    }
    if (!open(PAMCONF, ">/etc/pam.d/netophost")) {
        print "install pam configuration file: Unable to create configuration file\n";
        return 0;
    }
    print PAMCONF "#%PAM-1.0\n";
    print PAMCONF "auth       sufficient       ".$prefix."pam_unix2.so shadow nullok\n";
    print PAMCONF "auth       sufficient       ".$prefix."pam_unix.so shadow nullok\n";
    print PAMCONF "auth       required         ".$prefix."pam_unix_auth.so shadow nullok\n";
    print PAMCONF "account    sufficient       ".$prefix."pam_unix2.so\n";
    print PAMCONF "account    sufficient       ".$prefix."pam_unix.so\n";
    print PAMCONF "account    required         ".$prefix."pam_unix_acct.so\n";
    close(PAMCONF);
}

sub setup_logrotate()
{
    $logrotated_dir = "/etc/logrotate.d";
    if( ! -d $logrotated_dir ) {
        system("/usr/bin/mkdir -p $logrotated_dir");
        if( ! -d $logrotated_dir ) {
            die "Cannot create $logrotated_dir\n";
        }
    }

    $logrotate_src = "/opt/netop/host/configure/logrotate";
    $logrotate_dst = "$logrotated_dir/netophost";
    if(lc "$D2U_CMD" eq "dos2unix") {
        $cmd = "$D2U_CMD -n $logrotate_src $logrotate_dst >/dev/null";
    } else {
        # assumes fromdos
        $cmd = "$CAT_CMD $logrotate_src | $D2U_CMD > $logrotate_dst";
    }
    dbg($cmd);
    system($cmd);
    if( ($? == -1) or ($? >> 8 != 0) ) {
        die "Cannot copy $logrotate_src to $logrotate_dst\n";
    }

    $UNCONFIGURE_COMMANDS .= "\n";
    $UNCONFIGURE_COMMANDS .= "# unregister log rotation\n";
    $UNCONFIGURE_COMMANDS .= "unlink \"$logrotated_dir/netophost\n\";";
    $UNCONFIGURE_COMMANDS .= "\n";
}

sub main()
{
    my $serial      = undef;
    my $autoinstall = 0;
    my $noinitd     = 0;
    my $licensedto  = "";
    my $is_upgrade  = 0;
    my $usage =  "Usage: configure.pl <param>\n"
          ." where <param> is:\n"
          ." installation:\n"
          ."    [--autoinstall] : non-interactive, do not ask questions.\n"
          ."    [--noinitd]     : do not add service startup script to /etc/init.d\n"
          ."    [--serial    <serial>] : license serial (UK00900-XXXX...). \n"
          ."    [--upgrade]            : this is an upgrade from a previous version of the Host\n"
          ."    or:\n"
          ."    [--install_netophostxd]   : configure netophostxd and exit\n"
          ."    [--uninstall_netophostxd] : unconfigure netophostxd and exit\n"
          ."    [--enable_startup]        : enable runlevel startup and exit\n"
          ."    [--disable_startup]       : disable runlevel startup and exit\n"
          ."    [--query_startup]         : print runlevel startup info and exit\n"
          ."    [--enable_crle]           : (Solaris) ensure correct crle and exit.\n"
          ."    [--help]                  : print help message and exit.\n"
          ."    [--version]               : print version info and exit.\n";
    while ($arg = shift @ARGV) {
          if ($arg eq "--version") {
                print "$VERSION\n";
                exit;
          } elsif ($arg eq "--autoinstall") {
                $autoinstall = 1;
          } elsif ($arg eq "--install_netophostxd") {
            die "Must be root " if(! su_test());
                system_setup();
                setup_tools();
            die "Could not install netophostxd" if (($HOSTXD_WORKS eq "yes") and !install_netophostxd());
            print "-oOo- netophostxd is successfully configured -oOo-\n";
            if($NEED_RESTART != 0) {
                print "-oOo- Note: display manager needs to be restarted (or machine rebooted). -oOo-\n";
                print "-oOo-       for the changes to take full effect.                         -oOo-\n";
            }
            exit;
          } elsif ($arg eq "--uninstall_netophostxd") {
            die "Must be root " if(! su_test());
                system_setup();
                setup_tools();
            die "Could not uninstall netophostxd" if (($HOSTXD_WORKS eq "yes") and !uninstall_netophostxd());
            print "-oOo- netophostxd is successfully unconfigured -oOo-\n";
            if($NEED_RESTART != 0) {
                print "-oOo- Note: display manager needs to be restarted (or machine rebooted). -oOo-\n";
                print "-oOo-       for the changes to take full effect.                         -oOo-\n";
            }
            exit;
          } elsif ($arg eq "--noinitd") {
                $noinitd = 1;
          } elsif ($arg eq "--help") {
                print $usage;
                exit;
          } elsif ($arg eq "--serial") {
                # next token is not another option ?
                 if ( ($ARGV[0]) and ( "--" ne substr($ARGV[0],0,2) ) ) { $serial = shift @ARGV; } 
          } elsif ($arg eq "--upgrade") {
                $is_upgrade = 1;
          } elsif ($arg eq "--enable_startup") {
            die "Must be root " if (!su_test());
                $UNCONFIGURE_COMMANDS  = "";
                system_setup();
                setup_tools();
                enable_startup(1);
                return 0;
        } elsif ($arg eq "--enable_crle") {
            die "Must be root " if (!su_test());
                $UNCONFIGURE_COMMANDS  = "";
                system_setup();
                setup_tools();
            ensure_solaris_environment();
                return 0;
          } elsif ($arg eq "--disable_startup") {
                die "Must be root " if (!su_test());
                $UNCONFIGURE_COMMANDS  = "";
                system_setup();
                setup_tools();
                disable_startup();
                return 0;
          } elsif ($arg eq "--query_startup") {
                die "Must be root " if (!su_test());
                $UNCONFIGURE_COMMANDS  = "";
                system_setup();
                setup_tools();
                query_startup();
                return 0;
          } elsif ($arg eq "--pam") {
                setup_pam();
                return 0;
          } else {
                print "Error: unknown arg: '$arg'\n";
                die $usage;
          }
    }
    die "Must be root " if (!su_test());

    $UNCONFIGURE_COMMANDS  = "#!/usr/bin/perl -w\n";
    $UNCONFIGURE_COMMANDS .= "# unconfigure script for Netop Host\n";
    $UNCONFIGURE_COMMANDS .= "if(-e \"$INITD_NETOPHOSTD\"){\n"; # BEGIN BLOCK 1
    $UNCONFIGURE_COMMANDS .= "    \`$INITD_NETOPHOSTD stop\`;\n";

    system_setup();
    setup_tools();

    if ($SYSTEM eq "solaris") {
        if(!ensure_solaris_environment()) {
            exit 1;
        }
    }

    if (!check_license($serial,$licensedto,$autoinstall)) {
        #die "Licence check failed, Netop Host not operational.\n";
        host_installed_but_not_validated;
        
        if ($SIGNAL_UNINSTALL != 0) {
            $! = 56;
        }
        exit (0);
    }
    
    
    if (!$autoinstall && !has_portal_account() && !init_portal_account($is_upgrade)) {
        exit(0);
    }
    

    if (($HOSTXD_WORKS eq "yes") and !install_netophostxd()) {
        print STDERR "Installation of netophostxd failed, please install manually.\n";
    }

    if (!$noinitd) {
        copy_startup();
        enable_startup(0);
        system("cd /;$INITD_NETOPHOSTD start");
    }

    setup_pam();

    $UNCONFIGURE_COMMANDS .= "}\n";  # END BLOCK 1
    $UNCONFIGURE_COMMANDS .= "`perl /opt/netop/host/configure/configure.pl --uninstall_netophostxd`;\n";

    setup_logrotate();

    $UNCONFIGURE_COMMANDS .= "unlink \"/var/opt/netop/host/unconfigure.pl\";\n";
    #$UNCONFIGURE_COMMANDS .= "unlink \"$LICFILE\";\n";
    $UNCONFIGURE_COMMANDS .= "print \"Netop Host successfully unconfigured\\n\";\n";
     
    write_unconfigure();
    chmod 0744,$UNCONFIGURE_SCRIPT;
    dbg("Unconfigure script is written to $UNCONFIGURE_SCRIPT\n");
    host_installed_and_validated( $NEED_RESTART ); 
}

main;
exit;
__END__
